// Client App JavaScript
$(document).ready(function() {
    // Initialize the app
    initializeApp();
    
    // Event listeners
    setupEventListeners();
    
    // Load initial data
    loadDashboardData();
    loadRecentJobs();
});

function generateCarRegFromId(id) {
    const letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    const l1 = letters[id % 26];
    const l2 = letters[(id * 3) % 26];
    const num = (20 + (id % 80)).toString().padStart(2, '0');
    const l3 = letters[(id * 5) % 26];
    const l4 = letters[(id * 7) % 26];
    const l5 = letters[(id * 11) % 26];
    return `${l1}${l2}${num} ${l3}${l4}${l5}`;
}

function normalizeJob(job) {
    return {
        ...job,
        status: (job.status || '').toLowerCase() === 'quoted' ? 'invoiced' : (job.status || ''),
        invoiceUrl: job.invoiceUrl || 'https://www.w3.org/WAI/ER/tests/xhtml/testfiles/resources/pdf/dummy.pdf',
        carReg: job.carReg || generateCarRegFromId(job.id || 0),
        service: job.service || job.description || 'Wheel Service',
        amount: typeof job.amount === 'number' ? job.amount : (typeof job.quote === 'number' ? job.quote : undefined)
    };
}

function initializeApp() {
    // Check if user is logged in
    const user = localStorage.getItem('user');
    if (!user) {
        // Redirect to login (for now, just show demo data)
        console.log('No user logged in, showing demo data');
    }
}

function setupEventListeners() {
    // New job button
    $('#new-job-btn').click(function() {
        $('#new-job-modal').removeClass('hidden').addClass('fade-in');
    });
    $('#bottom-new-job-btn').click(function() {
        $('#new-job-modal').removeClass('hidden').addClass('fade-in');
    });
    
    // Close modal
    $('#cancel-job').click(function() {
        $('#new-job-modal').addClass('hidden');
        resetNewJobForm();
    });
    
    // Submit new job form
    $('#new-job-form').submit(function(e) {
        e.preventDefault();
        submitNewJob();
    });

    // New job back button
    $('#new-job-back').on('click', function() {
        $('#new-job-modal').addClass('hidden');
        resetNewJobForm();
    });

    // Customer auto-fill removed

    // Photo previews
    $('#wheel-photos').on('change', function(e) {
        const files = Array.from(e.target.files || []).slice(0, 10);
        const previews = $('#photo-previews');
        previews.empty();
        files.forEach(file => {
            const reader = new FileReader();
            reader.onload = function(evt) {
                const src = evt.target.result;
                previews.append(`<img src="${src}" alt="preview" class="photo-preview" />`);
            };
            reader.readAsDataURL(file);
        });
    });

    // Quantity tile interactions
    $(document).on('click', 'label:has(input[name="quantity"])', function() {
        $('label:has(input[name="quantity"])').removeClass('border-[#00a9ea]').addClass('border-gray-200');
        $(this).removeClass('border-gray-200').addClass('border-[#00a9ea]');
        $(this).find('input').prop('checked', true).trigger('change');
    });
    // Service tiles as radio
    $(document).on('click', 'label:has(input[name="service"])', function() {
        $('label:has(input[name="service"])').removeClass('border-[#00a9ea]').addClass('border-gray-200');
        $(this).removeClass('border-gray-200').addClass('border-[#00a9ea]');
        $(this).find('input').prop('checked', true).trigger('change');
        const val = $(this).find('input').val();
        if (val === 'other') {
            $('#job-description').focus();
        }
    });

    // UK reg plate formatting: uppercase and enforce pattern AA00 AAA with single space
    $('#reg-no').on('input', function() {
        let v = $(this).val().toUpperCase().replace(/[^A-Z0-9]/g, '');
        if (v.length > 4) {
            v = v.slice(0, 4) + ' ' + v.slice(4, 7);
        }
        $(this).val(v.slice(0, 8));
    });
    
    // View quotes button (removed for simplicity)
    

    // Bottom nav actions
    $('#nav-jobs').click(function() {
        openJobsModal();
    });
    $('#nav-invoices').click(function() {
        openInvoicesModal();
    });
    
    // Jobs modal
    $('#jobs-back').click(function() {
        closeJobsModal();
    });
    
    // Jobs search
    $('#jobs-search').on('input', function() {
        applyJobsFilters();
    });
    
    // Quick action tiles
    $('.quick-action-tile').click(function() {
        const service = $(this).data('service');
        openNewJobModalWithService(service);
    });

    // Orders toolbar interactions (removed - no longer needed)

    // Job card click -> open detail modal
    $(document).on('click', '.job-card', function() {
        const jobId = $(this).data('job-id');
        const all = window.__allJobs || [];
        const job = all.find(j => j.id === jobId);
        if (job) {
            openJobDetail(job);
        }
    });

    // Detail modal close
    $('#job-detail-dismiss, #job-detail-back').on('click', function() {
        $('#job-detail-modal').addClass('hidden');
        $('#job-detail-content').empty();
        // If jobs modal is open, keep it open (don't close it)
        // The jobs modal will remain visible behind the job detail modal
    });
}

function openNewJobModalWithService(service) {
    // Reset form first
    resetNewJobForm();
    
    // Open modal
    $('#new-job-modal').removeClass('hidden').addClass('fade-in');
    
    // Pre-select the service after a short delay to ensure modal is rendered
    if (service) {
        setTimeout(() => {
            // Map service names to radio button values
            const serviceMap = {
                'powder-coat': 'pc',
                'diamond-cut': 'dc',
                'other': 'other'
            };
            
            const serviceValue = serviceMap[service];
            if (serviceValue) {
                // First, unselect all service radio buttons and reset their visual state
                $('input[name="service"]').prop('checked', false);
                $('input[name="service"]').closest('label').removeClass('border-[#00a9ea]').addClass('border-gray-200');
                
                // Find and select the appropriate radio button
                $(`input[name="service"][value="${serviceValue}"]`).prop('checked', true);
                
                // Update the visual state of the radio tile
                $(`input[name="service"][value="${serviceValue}"]`).closest('label').addClass('border-[#00a9ea]').removeClass('border-gray-200');
                
                // If "other" is selected, focus the job description field
                if (serviceValue === 'other') {
                    setTimeout(() => {
                        $('#job-description').focus();
                    }, 100);
                }
            }
        }, 100);
    }
}

function loadDashboardData() {
    // Simulate loading dashboard data
    $.ajax({
        url: 'data/dashboard.json',
        method: 'GET',
        dataType: 'json',
        success: function(data) {
            updateDashboard(data);
        },
        error: function() {
            // Fallback to demo data
            const demoData = {
                activeJobs: 5,
                pendingQuotes: 2,
                completedJobs: 12
            };
            updateDashboard(demoData);
        }
    });
}

function updateDashboard(data) {
    $('#active-jobs').text(data.activeJobs);
    $('#pending-quotes').text(data.pendingQuotes);
    $('#completed-jobs').text(data.completedJobs);
}

function loadRecentJobs() {
    $.ajax({
        url: 'data/jobs.json',
        method: 'GET',
        dataType: 'json',
        success: function(data) {
            const normalized = (data.jobs || []).map(job => normalizeJob({
                ...job,
                invoiceUrl: job.invoiceUrl || '../shared-resources/pdfs/sample-invoice.pdf'
            }));
            window.__allJobs = normalized;
        },
        error: function() {
            // Fallback to demo data
            const demoJobs = [
                {
                    id: 1,
                    description: "BMW 18\" Alloy Wheel Refurbishment",
                    status: "in-progress",
                    date: "2024-01-15",
                    quantity: 4
                },
                {
                    id: 2,
                    description: "Mercedes 19\" Diamond Cut Wheels",
                    status: "invoiced",
                    date: "2024-01-14",
                    quantity: 4,
                    invoiceUrl: '../shared-resources/pdfs/sample-invoice.pdf'
                },
                {
                    id: 3,
                    description: "Audi 17\" Standard Refurbishment",
                    status: "completed",
                    date: "2024-01-12",
                    quantity: 4
                }
            ];
            window.__allJobs = demoJobs.map(normalizeJob);
        }
    });
}

function createJobElement(job) {
    const statusClass = getStatusClass(job.status);
    const statusText = getStatusText(job.status);
    const statusIcon = getStatusIcon(job.status);
    
    return $(`
        <div class="relative bg-white rounded-2xl shadow-xl border border-gray-100 p-4 py-6 hover:shadow-2xl transition-shadow cursor-pointer job-card" data-job-id="${job.id}">
            <span class="absolute -top-3 left-3 inline-flex items-center px-3 py-1 text-xs rounded-full border ${statusClass} z-10">
                <i class="${statusIcon} mr-2"></i>
                ${statusText}
            </span>
            <div class="flex items-start justify-between">
                <div class="flex items-center space-x-4">
                    <div class="w-12 h-12  rounded-xl flex items-center justify-center shadow-lg">
                        <svg width="40px" height="40px" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><g id="SVGRepo_bgCarrier" stroke-width="0"></g><g id="SVGRepo_tracerCarrier" stroke-linecap="round" stroke-linejoin="round"></g><g id="SVGRepo_iconCarrier"> <path d="M21 12C21 17.5228 18.9853 22 16.5 22M16.5 2C18.3453 2 19.9312 4.46819 20.6256 8" stroke="#1C274C" stroke-width="1.5" stroke-linecap="round"></path> <path d="M3 12C3 6.47715 5.01472 2 7.5 2C9.98528 2 12 6.47715 12 12C12 17.5228 9.98528 22 7.5 22C5.65471 22 4.06884 19.5318 3.37444 16" stroke="#1C274C" stroke-width="1.5" stroke-linecap="round"></path> <path d="M7.5 22C9.98528 22 12 17.5228 12 12C12 6.47715 9.98528 2 7.5 2" stroke="#1C274C" stroke-width="1.5"></path> <path d="M9 12C9 15.3137 8.32843 18 7.5 18C6.67157 18 6 15.3137 6 12C6 8.68629 6.67157 6 7.5 6C8.32843 6 9 8.68629 9 12Z" stroke="#1C274C" stroke-width="1.5"></path> <path d="M7.5 2L16.5 2" stroke="#1C274C" stroke-width="1.5"></path> <path d="M7.5 22L16.5 22" stroke="#1C274C" stroke-width="1.5"></path> <path d="M9 12H8" stroke="#1C274C" stroke-width="1.5" stroke-linecap="round"></path> </g></svg>
                    </div>
                    <div>
                        <div class="flex items-center flex-wrap gap-2 mb-1">
                            <span class="inline-flex items-center px-2 py-1 rounded-lg bg-[#ffd900] text-gray-800 text-xs font-semibold tracking-wide">${job.carReg}</span>
                            <span class="font-bold text-gray-900 text-sm md:text-lg">${job.service}</span>
                            </div>
                        <div class="flex flex-wrap gap-4 text-sm text-gray-600">
                            <span class="flex items-center"><i class="fas fa-cog text-[#00a9ea] mr-2"></i>${job.quantity} wheels</span>
                            
                            <span class="flex items-center"><i class="fas fa-calendar text-[#00a9ea] mr-2"></i>${formatDate(job.date)}</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `);
}

function showJobsSkeleton() {
    const container = $('#recent-jobs-list');
    $('#orders-empty').addClass('hidden');
    container.empty();
    for (let i = 0; i < 6; i++) {
        container.append(`
            <div class="skeleton-card">
                <div class="flex items-start justify-between">
                    <div class="flex items-center space-x-4 w-full">
                        <div class="w-12 h-12 skeleton rounded-xl"></div>
                        <div class="flex-1">
                            <div class="skeleton skeleton-line w-2/3 mb-3"></div>
                            <div class="flex gap-4">
                                <div class="skeleton skeleton-line w-16"></div>
                                <div class="skeleton skeleton-line w-24"></div>
                                <div class="skeleton skeleton-line w-20"></div>
                            </div>
                        </div>
                    </div>
                    <div class="skeleton w-24 h-8 rounded-xl"></div>
                </div>
            </div>
        `);
    }
}

function applyJobsFilters() {
    // Check if we're in jobs modal or home page
    const isJobsModal = !$('#jobs-modal').hasClass('hidden');
    const query = isJobsModal ? 
        ($('#jobs-search').val() || '').toLowerCase() : 
        ($('#orders-search').val() || '').toLowerCase();
    
    const all = window.__allJobs || [];
    let filtered = all.slice();

    if (query) {
        filtered = filtered.filter(j =>
            String(j.id).includes(query) ||
            (j.description || '').toLowerCase().includes(query) ||
            (j.carReg || '').toLowerCase().includes(query) ||
            (j.service || '').toLowerCase().includes(query)
        );
    }

    // Sort: in-progress first, then pending, invoiced, completed, then by date desc
    const statusOrder = { 'in-progress': 0, 'pending': 1, 'invoiced': 2, 'completed': 3 };
    filtered.sort((a, b) => {
        const sa = statusOrder[(a.status || '').toLowerCase()] ?? 99;
        const sb = statusOrder[(b.status || '').toLowerCase()] ?? 99;
        if (sa !== sb) return sa - sb;
        const da = new Date(a.date || 0).getTime();
        const db = new Date(b.date || 0).getTime();
        return db - da;
    });

    if (isJobsModal) {
        displayJobsInModal(filtered);
    } else {
        const container = $('#recent-jobs-list');
        container.empty();

        if (!filtered.length) {
            $('#orders-empty').removeClass('hidden');
            return;
        } else {
            $('#orders-empty').addClass('hidden');
        }

        filtered.forEach(job => container.append(createJobElement(job)));
    }
}

function getStatusClass(status) {
    const statusMap = {
        'pending': 'bg-amber-50 text-amber-700 border-amber-200',
        'in-progress': 'bg-[#00a9ea]/10 text-[#00a9ea] border-[#00a9ea]/20',
        'completed': 'bg-emerald-50 text-emerald-700 border-emerald-200',
        'invoiced': 'bg-violet-50 text-violet-700 border-violet-200'
    };
    return statusMap[(status || '').toLowerCase()] || 'bg-amber-50 text-amber-700 border-amber-200';
}

function getStatusText(status) {
    const statusMap = {
        'pending': 'Pending',
        'in-progress': 'In Progress',
        'completed': 'Completed',
        'invoiced': 'Invoiced'
    };
    return statusMap[(status || '').toLowerCase()] || 'Pending';
}

function getStatusIcon(status) {
    const iconMap = {
        'pending': 'fas fa-clock',
        'in-progress': 'fas fa-tools',
        'completed': 'fas fa-check-circle',
        'invoiced': 'fas fa-file-invoice-dollar'
    };
    return iconMap[(status || '').toLowerCase()] || 'fas fa-clock';
}

function formatDate(dateString) {
    const date = new Date(dateString);
    if (isNaN(date)) return '';
    const day = date.getDate();
    const month = date.toLocaleString('en-GB', { month: 'short' });
    const year2 = String(date.getFullYear()).slice(-2);
    return `${day} ${month}, ${year2}`;
}

function openJobDetail(job) {
    $('#job-detail-title').html(`
        <div class="flex items-center flex-wrap gap-2">
            <span class="inline-flex items-center px-2 py-1 rounded-lg bg-[#ffd900] text-gray-900 text-xs font-bold tracking-wide">${job.carReg || ''}</span>
            <span class="font-bold text-[#00a9ea] text-lg">${job.service}</span>
        </div>
    `);
    $('#job-detail-subtitle').html(`
        <span class="mr-3">${formatDate(job.date)}</span>
        <span class="mr-3">${job.quantity} wheels</span>
        ${typeof job.amount === 'number' ? `<span class="mr-3">£${job.amount.toFixed(2)}</span>` : ''}
        ${job.completionDate ? `<span class=\"mr-3 text-emerald-700\">Completed: ${formatDate(job.completionDate)}</span>` : ''}
    `);

    const statusClass = getStatusClass(job.status);
    const statusText = getStatusText(job.status);
    const statusIcon = getStatusIcon(job.status);

    const content = $(`
        <div class="space-y-6">
            <div class="flex items-center justify-between">
                <div class="text-sm text-gray-600">
                    <div class="mb-1"><span class="font-semibold text-gray-800">Status:</span></div>
                </div>
                <span class="inline-flex items-center px-3 py-1.5 text-xs font-semibold rounded-full border ${statusClass}">
                    <i class="${statusIcon} mr-2"></i>${statusText}
                </span>
            </div>
            <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 text-sm">
                <div class="bg-gray-50 rounded-xl p-4 border border-gray-100">
                    <div class="text-gray-500 mb-1">Job ID</div>
                    <div class="font-semibold text-gray-900">#${job.id}</div>
                </div>
                <div class="bg-gray-50 rounded-xl p-4 border border-gray-100">
                    <div class="text-gray-500 mb-1">Created</div>
                    <div class="font-semibold text-gray-900">${formatDate(job.date)}</div>
                </div>
                <div class="bg-gray-50 rounded-xl p-4 border border-gray-100">
                    <div class="text-gray-500 mb-1">Quantity</div>
                    <div class="font-semibold text-gray-900">${job.quantity}</div>
                </div>
                <div class="bg-gray-50 rounded-xl p-4 border border-gray-100">
                    <div class="text-gray-500 mb-1">Service</div>
                    <div class="font-semibold text-gray-900">${job.service || '—'}</div>
                </div>
                ${job.completionDate ? `
                <div class="bg-emerald-50 rounded-xl p-4 border border-emerald-200 sm:col-span-2">
                    <div class="text-emerald-700 mb-1 font-semibold">Completed</div>
                    <div class="font-semibold text-emerald-800">${formatDate(job.completionDate)}</div>
                </div>
                ` : ''}
            </div>
            <div>
                <div class="text-sm text-gray-500 mb-2">Description</div>
                <div class="text-gray-800">${job.description}</div>
            </div>
            ${Array.isArray(job.addons) && job.addons.length ? `
            <div>
                <div class="text-sm text-gray-500 mb-2">Add-ons</div>
                <div class="flex flex-wrap gap-2">
                    ${job.addons.map(a => `<span class=\"px-3 py-1 rounded-full border text-sm bg-gray-50 border-gray-200 text-gray-700\">${typeof a === 'string' ? a : (a.name || 'Addon')}</span>`).join('')}
                </div>
            </div>
            ` : ''}
            ${((job.status || '').toLowerCase() === 'invoiced') ? `
            <div class="pt-2">
                <div class="flex items-center justify-between mb-3">
                    <div class="text-sm text-gray-500">Invoice</div>
                    <a href="${job.invoiceUrl || '#'}" target="_blank" class="text-[#00a9ea] text-sm font-semibold hover:underline">Download PDF</a>
                </div>
                <div class="rounded-xl overflow-hidden border border-gray-200">
                    <iframe src="${job.invoiceUrl || ''}" class="w-full" style="height: 420px;" title="Invoice PDF"></iframe>
                </div>
            </div>
            ` : ''}
        </div>
    `);

    $('#job-detail-content').empty().append(content);
    $('#job-detail-modal').removeClass('hidden').addClass('fade-in');
}

function openJobsModal() {
    $('#jobs-modal').removeClass('hidden').addClass('fade-in');
    loadJobsForModal();
}

function closeJobsModal() {
    $('#jobs-modal').addClass('hidden').removeClass('fade-in');
}

function loadJobsForModal() {
    // Use global jobs data if available, otherwise load it
    if (window.__allJobs && window.__allJobs.length > 0) {
        displayJobsInModal(window.__allJobs);
        return;
    }
    
    // Show loading state
    $('#jobs-list').html(`
        <div class="col-span-full flex justify-center items-center py-12">
            <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-[#00a9ea]"></div>
        </div>
    `);
    
    // Load jobs data
    $.ajax({
        url: 'data/jobs.json',
        method: 'GET',
        dataType: 'json',
        success: function(data) {
            const normalized = (data.jobs || []).map(job => normalizeJob({
                ...job,
                invoiceUrl: job.invoiceUrl || '../shared-resources/pdfs/sample-invoice.pdf'
            }));
            window.__allJobs = normalized;
            displayJobsInModal(normalized);
        },
        error: function() {
            // Fallback to demo data
            const demoJobs = [
                {
                    id: 1,
                    description: "BMW 18\" Alloy Wheel Refurbishment",
                    status: "in-progress",
                    date: "2024-01-15",
                    quantity: 4
                },
                {
                    id: 2,
                    description: "Mercedes 19\" Diamond Cut Wheels",
                    status: "invoiced",
                    date: "2024-01-14",
                    quantity: 4,
                    invoiceUrl: '../shared-resources/pdfs/sample-invoice.pdf'
                },
                {
                    id: 3,
                    description: "Audi 17\" Standard Refurbishment",
                    status: "completed",
                    date: "2024-01-12",
                    quantity: 4
                }
            ];
            const normalized = demoJobs.map(job => normalizeJob(job));
            window.__allJobs = normalized;
            displayJobsInModal(normalized);
        }
    });
}

function displayJobsInModal(jobs) {
    const container = $('#jobs-list');
    
    if (jobs.length === 0) {
        container.html(`
            <div class="col-span-full text-center py-12">
                <div class="w-16 h-16 mx-auto mb-4 rounded-2xl bg-gray-100 flex items-center justify-center">
                    <i class="fas fa-box-open text-gray-500"></i>
                </div>
                <p class="font-semibold text-gray-800">No jobs found</p>
                <p class="text-sm text-gray-600">Try searching or create a new job.</p>
            </div>
        `);
        return;
    }
    
    container.empty();
    jobs.forEach(job => {
        const jobElement = createJobElement(job);
        container.append(jobElement);
    });
}

function openInvoicesModal() {
    const invoices = [
        { id: 'INV-1001', jobId: 2, date: '2024-01-14', amount: 480.00, url: '../shared-resources/pdfs/sample-invoice.pdf' },
        { id: 'INV-1002', jobId: 5, date: '2024-01-15', amount: 650.00, url: '../shared-resources/pdfs/sample-invoice.pdf' },
        { id: 'INV-1003', jobId: 3, date: '2024-01-16', amount: 280.00, url: '../shared-resources/pdfs/sample-invoice.pdf' }
    ];

    const list = $('#invoices-list');
    list.empty();
    invoices.forEach(inv => {
        const row = $(`
            <div class="flex items-center justify-between px-4 py-4">
                <div class="flex items-center gap-4">
                    <div class="w-10 h-10 bg-gradient-to-br from-[#00a9ea] to-[#0088cc] rounded-xl flex items-center justify-center text-white shadow">
                        <i class="fas fa-file-invoice-dollar"></i>
                    </div>
                    <div>
                        <div class="font-semibold text-gray-900">${inv.id}</div>
                        <div class="text-sm text-gray-500">Job #${inv.jobId} • ${formatDate(inv.date)}</div>
                    </div>
                </div>
                <div class="flex items-center gap-4">
                    <div class="font-semibold text-gray-900">£${inv.amount.toFixed(2)}</div>
                    <button class="px-3 py-2 rounded-xl bg-gray-100 text-gray-700 text-sm font-semibold hover:bg-gray-200 view-invoice" data-id="${inv.id}">View</button>
                </div>
            </div>
        `);
        row.find('.view-invoice').on('click', function() {
            $('#invoices-list-view').addClass('hidden');
            $('#invoice-viewer').removeClass('hidden');
            const pdfUrl = inv.url || '../shared-resources/pdfs/sample-invoice.pdf';
            $('#invoice-iframe').attr('src', pdfUrl);
            $('#invoice-download').attr('href', pdfUrl);
            $('#invoice-viewer-title').text(inv.id);
            $('#invoice-viewer-subtitle').text(`Job #${inv.jobId} • ${formatDate(inv.date)} • £${inv.amount.toFixed(2)}`);
        });
        list.append(row);
    });

    $('#invoices-modal').removeClass('hidden').addClass('fade-in');

    $('#invoice-back').off('click').on('click', function() {
        $('#invoice-iframe').attr('src', '');
        $('#invoice-viewer').addClass('hidden');
        $('#invoices-list-view').removeClass('hidden');
    });

    // Back button in mobile modal
    $('#invoices-back').off('click').on('click', function() {
        $('#invoice-iframe').attr('src', '');
        $('#invoice-viewer').addClass('hidden');
        $('#invoices-list-view').removeClass('hidden');
        $('#invoices-modal').addClass('hidden');
    });
}

function submitNewJob() {
    const formData = {
        description: $('#job-description').val(),
        regNo: $('#reg-no').val(),
        quantity: Number($("input[name='quantity']:checked").val() || 0),
        service: $("input[name='service']:checked").val() || '',
        // urgency removed,
        photos: $('#wheel-photos')[0].files
    };
    
    // Validate form
    if (!validateJobForm(formData)) {
        return;
    }
    
    // Show loading
    const submitBtn = $('#new-job-form button[type="submit"]');
    const originalText = submitBtn.text();
    submitBtn.html('<span class="spinner"></span> Submitting...');
    submitBtn.prop('disabled', true);
    
    // No customer profile to save
    
    // Simulate API call
    setTimeout(() => {
        // Show confirmation screen
        $('#new-job-form').addClass('hidden');
        $('#new-job-confirmation').removeClass('hidden').addClass('fade-in');

        // Reload dashboard/jobs in background
        loadDashboardData();
        loadRecentJobs();
        
        // Reset button
        submitBtn.text(originalText);
        submitBtn.prop('disabled', false);
    }, 1200);
}

function validateJobForm(data) {
    let isValid = true;
    
    // Clear previous errors
    $('.error').removeClass('error');
    $('.error-message').remove();
    
    if (!data.quantity || data.quantity < 1) {
        showFieldError('#quantity-options', 'Quantity must be at least 1');
        isValid = false;
    }
    
    if (!data.service) {
        showFieldError('#service-options', 'Please select a service');
        isValid = false;
    }
    
    // Customer details removed
    
    return isValid;
}

function showFieldError(fieldSelector, message) {
    const $field = $(fieldSelector);
    $field.addClass('error border-red-500 bg-red-50');
    // Remove any existing error message adjacent to avoid duplicates
    if ($field.next('.error-message').length) {
        $field.next('.error-message').remove();
    }
    $field.after(`<div class="error-message text-red-600 text-sm mt-2 font-semibold">${message}</div>`);
}

function resetNewJobForm() {
    $('#new-job-form')[0].reset();
    $('.error').removeClass('error border-red-500 bg-red-50');
    $('.error-message').remove();
    $('#photo-previews').empty();
    $('#new-job-form').removeClass('hidden');
    $('#new-job-confirmation').addClass('hidden');
}

// Quotes functionality removed for simplicity

function showNotification(message, type = 'info') {
    const notification = $(`
        <div class="notification notification-${type}">
            ${message}
        </div>
    `);
    
    $('body').append(notification);
    
    // Auto remove after 3 seconds
    setTimeout(() => {
        notification.fadeOut(() => {
            notification.remove();
        });
    }, 3000);
}

function logout() {
    localStorage.removeItem('user');
    showNotification('Logged out successfully', 'info');
    
    // In a real app, redirect to login page
    setTimeout(() => {
        window.location.href = 'login.html';
    }, 1000);
}
