# Xero PHP SDK Setup Guide

## Prerequisites

### 1. Install Composer
- Download from: https://getcomposer.org/
- Install globally on your system
- Verify installation: `composer --version`

### 2. PHP Requirements
- PHP 7.4 or higher
- cURL extension enabled
- JSON extension enabled
- OpenSSL extension enabled

## Installation Steps

### Step 1: Install Xero SDK
Run the installation script:
```bash
# Windows
install-xero-sdk.bat

# Or manually
composer install
```

### Step 2: Get Xero Credentials
1. Go to [developer.xero.com](https://developer.xero.com)
2. Sign in with your Xero account
3. Click "New App" or "Add App"
4. Fill in app details:
   - **App Name**: RefurbFlow Ordering System
   - **App URL**: https://localhost:8080/in-house-app
   - **Redirect URI**: https://localhost:8080/in-house-app/xero-sdk-api.php?action=authorize
5. Note down your **Client ID** and **Client Secret**

### Step 3: Configure Credentials
Update `xero-sdk-api.php` with your credentials:
```php
$XERO_CLIENT_ID = 'YOUR_CLIENT_ID';
$XERO_CLIENT_SECRET = 'YOUR_CLIENT_SECRET';
$XERO_REDIRECT_URI = 'https://localhost:8080/in-house-app/xero-sdk-api.php?action=authorize';
```

### Step 4: Authorize the App
1. Go to: `https://login.xero.com/identity/connect/authorize?response_type=code&client_id=YOUR_CLIENT_ID&redirect_uri=https://localhost:8080/in-house-app/xero-sdk-api.php?action=authorize&scope=accounting.transactions%20accounting.contacts.read%20accounting.settings.read&state=xero_auth_state`
2. Authorize the app
3. You'll be redirected back with an authorization code
4. The system will automatically exchange the code for tokens

### Step 5: Test Integration
1. Go to any job in your system
2. Click "Ready" button
3. Check your Xero account for the created invoice

## File Structure

```
in-house-app/
├── composer.json              # Composer configuration
├── composer.lock              # Lock file (generated)
├── vendor/                    # Xero SDK (generated)
├── xero-sdk-api.php          # New SDK-based API
├── xero_tokens.json          # Stored tokens (generated)
├── install-xero-sdk.bat      # Installation script
└── XERO_SDK_SETUP.md         # This guide
```

## API Endpoints

- `xero-sdk-api.php?action=auth_url` - Get authorization URL
- `xero-sdk-api.php?action=authorize` - Handle OAuth2 callback
- `xero-sdk-api.php?action=create_invoice` - Create invoice using SDK
- `xero-sdk-api.php?action=test_connection` - Test connection

## Benefits of Using SDK

### 1. Official Support
- Maintained by Xero team
- Regular updates and bug fixes
- Full API coverage

### 2. Better Error Handling
- Proper exception handling
- Detailed error messages
- Automatic retry logic

### 3. Type Safety
- Strongly typed objects
- IDE autocomplete support
- Better code documentation

### 4. OAuth2 Management
- Automatic token refresh
- Secure token storage
- Proper authorization flow

## Troubleshooting

### Common Issues

#### 1. Composer Not Found
```
ERROR: Composer is not installed or not in PATH
```
**Solution**: Install Composer and add it to your system PATH

#### 2. PHP Version Error
```
ERROR: This package requires PHP 7.4 or higher
```
**Solution**: Update PHP to version 7.4 or higher

#### 3. Missing Extensions
```
ERROR: cURL extension not found
```
**Solution**: Enable required PHP extensions in php.ini

#### 4. Authorization Failed
```
ERROR: No valid tokens found
```
**Solution**: Re-authorize the app using the authorization URL

### Debug Mode

Enable detailed logging by checking your web server error logs:
- Look for `=== XERO SDK INTEGRATION ATTEMPT ===`
- Check for `SUCCESS:` or `ERROR:` messages
- Verify token storage in `xero_tokens.json`

## Security Notes

### 1. Token Storage
- Tokens are stored in `xero_tokens.json`
- This file should be protected (not web-accessible)
- Consider encrypting tokens for production

### 2. Credentials
- Never commit credentials to version control
- Use environment variables for production
- Rotate credentials regularly

### 3. HTTPS
- Always use HTTPS in production
- Update redirect URIs accordingly
- Verify SSL certificates

## Production Deployment

### 1. Environment Variables
```php
$XERO_CLIENT_ID = getenv('XERO_CLIENT_ID');
$XERO_CLIENT_SECRET = getenv('XERO_CLIENT_SECRET');
```

### 2. Token Encryption
```php
// Encrypt tokens before storing
$encryptedTokens = encrypt(json_encode($tokens));
file_put_contents('xero_tokens.json', $encryptedTokens);
```

### 3. Error Monitoring
- Set up proper error logging
- Monitor API rate limits
- Implement retry mechanisms

## Support

- **Xero Developer Portal**: https://developer.xero.com
- **SDK Documentation**: https://github.com/XeroAPI/xero-php-oauth2
- **API Reference**: https://developer.xero.com/documentation/api/accounting/overview
