<?php
// Debug contact creation issue
echo "=== Debugging Contact Creation Issue ===\n\n";

require_once 'vendor/autoload.php';

use XeroAPI\XeroPHP\Api\AccountingApi;
use XeroAPI\XeroPHP\Configuration;
use XeroAPI\XeroPHP\Models\Accounting\Contact;
use XeroAPI\XeroPHP\Models\Accounting\Address;
use XeroAPI\XeroPHP\Models\Accounting\Phone;
use XeroAPI\XeroPHP\Models\Accounting\Contacts;
use GuzzleHttp\Client;

// Load tokens
$tokens = json_decode(file_get_contents('xero_tokens.json'), true);
$accessToken = $tokens['access_token'];
$tenantId = $tokens['tenant_id'];

// Configure Xero API
$config = Configuration::getDefaultConfiguration();
$config->setAccessToken($accessToken);
$config->setHost('https://api.xero.com');

$guzzleClient = new Client([
    'verify' => false,
    'timeout' => 30
]);

$apiInstance = new AccountingApi($guzzleClient, $config);

try {
    // Create a test contact with minimal details first
    $contact = new Contact();
    $contact->setName('Debug Test Insurance Ltd');
    
    echo "1. Creating minimal contact first...\n";
    
    $contacts = new Contacts();
    $contacts->setContacts([$contact]);
    
    $result = $apiInstance->createContacts($tenantId, $contacts);
    
    if ($result->getContacts() && count($result->getContacts()) > 0) {
        $createdContact = $result->getContacts()[0];
        echo "   ✅ Minimal contact created: " . $createdContact->getContactID() . "\n";
        
        // Now try to update it with full details
        echo "\n2. Updating contact with full details...\n";
        
        $contactId = $createdContact->getContactID();
        
        // Get the contact first
        $existingContact = $apiInstance->getContact($tenantId, $contactId);
        
        if ($existingContact->getContacts() && count($existingContact->getContacts()) > 0) {
            $contactToUpdate = $existingContact->getContacts()[0];
            
            // Add address
            $address = new Address();
            $address->setAddressType('STREET');
            $address->setAddressLine1('456 Debug Street');
            $address->setCity('Manchester');
            $address->setPostalCode('M1 1AA');
            $address->setCountry('United Kingdom');
            $contactToUpdate->setAddresses([$address]);
            
            // Add phone
            $phone = new Phone();
            $phone->setPhoneType('DEFAULT');
            $phone->setPhoneNumber('0800 999 888');
            $contactToUpdate->setPhones([$phone]);
            
            // Add email
            $contactToUpdate->setEmailAddress('debug@insurance.com');
            
            echo "   Setting address: " . $address->getAddressLine1() . "\n";
            echo "   Setting phone: " . $phone->getPhoneNumber() . "\n";
            echo "   Setting email: " . $contactToUpdate->getEmailAddress() . "\n";
            
            // Update the contact
            $updateContacts = new Contacts();
            $updateContacts->setContacts([$contactToUpdate]);
            
            $updateResult = $apiInstance->updateContact($tenantId, $contactId, $updateContacts);
            
            if ($updateResult->getContacts() && count($updateResult->getContacts()) > 0) {
                $updatedContact = $updateResult->getContacts()[0];
                echo "\n3. Updated contact details:\n";
                echo "   Name: " . $updatedContact->getName() . "\n";
                echo "   Email: " . ($updatedContact->getEmailAddress() ?? 'Not set') . "\n";
                echo "   Phone: " . (count($updatedContact->getPhones()) > 0 ? $updatedContact->getPhones()[0]->getPhoneNumber() : 'Not set') . "\n";
                
                $addresses = $updatedContact->getAddresses();
                if (!empty($addresses)) {
                    $addr = $addresses[0];
                    echo "   Address: " . $addr->getAddressLine1() . "\n";
                    echo "   City: " . $addr->getCity() . "\n";
                    echo "   Postcode: " . $addr->getPostalCode() . "\n";
                    echo "   Country: " . $addr->getCountry() . "\n";
                } else {
                    echo "   Address: Not set\n";
                }
            } else {
                echo "   ❌ Update failed - no contacts returned\n";
            }
        } else {
            echo "   ❌ Could not retrieve contact for update\n";
        }
    } else {
        echo "   ❌ Minimal contact creation failed\n";
    }
    
} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
    echo "   File: " . $e->getFile() . "\n";
    echo "   Line: " . $e->getLine() . "\n";
}

echo "\n=== Debug Complete ===\n";
?>
