<?php
// Debug invoice creation to see what's really happening
echo "=== Debug Invoice Creation ===\n\n";

// Load stored tokens
$tokensFile = 'xero_tokens.json';
if (!file_exists($tokensFile)) {
    echo "❌ No tokens found. Please authorize the app first.\n";
    exit;
}

$tokens = json_decode(file_get_contents($tokensFile), true);
if (!$tokens || !isset($tokens['access_token'])) {
    echo "❌ Invalid token file. Please re-authorize the app.\n";
    exit;
}

$accessToken = $tokens['access_token'];
$tenantId = $tokens['tenant_id'];

echo "1. Current tokens:\n";
echo "   Access Token: " . substr($accessToken, 0, 50) . "...\n";
echo "   Tenant ID: $tenantId\n";
echo "   Expires: " . date('Y-m-d H:i:s', $tokens['created_at'] + $tokens['expires_in']) . "\n\n";

// Check if token is expired
$now = time();
$expiresAt = $tokens['created_at'] + $tokens['expires_in'];
if ($now > $expiresAt) {
    echo "❌ TOKEN EXPIRED! Need to re-authorize.\n";
    echo "   Current time: " . date('Y-m-d H:i:s', $now) . "\n";
    echo "   Expires at: " . date('Y-m-d H:i:s', $expiresAt) . "\n";
    exit;
} else {
    echo "✅ Token is still valid\n\n";
}

echo "2. Testing invoice creation with detailed debugging...\n";

// Sample invoice data
$invoiceData = [
    'jobId' => 888,
    'customer' => 'Debug Test Customer',
    'description' => 'Debug Test Invoice - ' . date('Y-m-d H:i:s'),
    'amount' => 100.00,
    'service' => 'Debug Test',
    'regNo' => 'DEBUG123'
];

echo "   Invoice Data: " . json_encode($invoiceData, JSON_PRETTY_PRINT) . "\n\n";

// Make the API call
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'http://localhost/mvps/RefurbFlow-ordering-system/in-house-app/xero-sdk-api.php?action=create_invoice');
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['invoiceData' => $invoiceData]));
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json'
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

echo "3. API Response:\n";
echo "   HTTP Code: $httpCode\n";
if ($error) {
    echo "   cURL Error: $error\n";
} else {
    echo "   Response: $response\n\n";
    
    $result = json_decode($response, true);
    if ($result) {
        echo "4. Parsed Response:\n";
        echo json_encode($result, JSON_PRETTY_PRINT) . "\n\n";
        
        if (isset($result['InvoiceID'])) {
            echo "✅ Invoice creation reported successful\n";
            echo "   Invoice ID: " . $result['InvoiceID'] . "\n";
            echo "   Invoice Number: " . $result['InvoiceNumber'] . "\n";
            echo "   Status: " . $result['Status'] . "\n\n";
            
            // Now immediately check if it exists in Xero
            echo "5. Immediately checking if invoice exists in Xero...\n";
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'https://api.xero.com/api.xro/2.0/Invoices');
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken,
                'Accept: application/json',
                'Xero-tenant-id: ' . $tenantId
            ]);
            
            $xeroResponse = curl_exec($ch);
            $xeroHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $xeroError = curl_error($ch);
            curl_close($ch);
            
            echo "   Xero API Response Code: $xeroHttpCode\n";
            if ($xeroError) {
                echo "   Xero API Error: $xeroError\n";
            } else {
                $xeroData = json_decode($xeroResponse, true);
                if (isset($xeroData['Invoices'])) {
                    $invoices = $xeroData['Invoices'];
                    echo "   Found " . count($invoices) . " invoices in Xero\n";
                    
                    // Look for our invoice
                    $found = false;
                    foreach ($invoices as $invoice) {
                        if (($invoice['InvoiceID'] ?? '') === $result['InvoiceID']) {
                            echo "   ✅ FOUND! Invoice exists in Xero\n";
                            echo "   Invoice Number: " . ($invoice['InvoiceNumber'] ?? 'Unknown') . "\n";
                            echo "   Status: " . ($invoice['Status'] ?? 'Unknown') . "\n";
                            $found = true;
                            break;
                        }
                    }
                    
                    if (!$found) {
                        echo "   ❌ NOT FOUND! Invoice doesn't exist in Xero\n";
                        echo "   This suggests the invoice creation is failing silently\n";
                    }
                }
            }
        } else {
            echo "❌ Invoice creation failed\n";
            echo "   Error: " . ($result['error'] ?? 'Unknown error') . "\n";
        }
    }
}

echo "\n=== Debug Complete ===\n";
?>
