<?php
// Debug Xero invoice creation
echo "=== Debugging Xero Invoice Creation ===\n\n";

require_once 'vendor/autoload.php';

use XeroAPI\XeroPHP\Api\AccountingApi;
use XeroAPI\XeroPHP\Configuration;
use XeroAPI\XeroPHP\Models\Accounting\Contact;
use XeroAPI\XeroPHP\Models\Accounting\Invoice;
use XeroAPI\XeroPHP\Models\Accounting\LineItem;
use XeroAPI\XeroPHP\Models\Accounting\Address;
use XeroAPI\XeroPHP\Models\Accounting\Phone;
use GuzzleHttp\Client;

// Check if we have stored tokens
if (!file_exists('xero_tokens.json')) {
    echo "❌ No Xero tokens found. Please authorize first.\n";
    exit;
}

$tokens = json_decode(file_get_contents('xero_tokens.json'), true);
$accessToken = $tokens['access_token'] ?? null;
$tenantId = $tokens['tenant_id'] ?? null;

if (!$accessToken || !$tenantId) {
    echo "❌ Invalid tokens. Please re-authorize.\n";
    exit;
}

echo "1. Tokens loaded successfully\n";
echo "   Access Token: " . substr($accessToken, 0, 20) . "...\n";
echo "   Tenant ID: $tenantId\n\n";

try {
    // Configure Xero API
    $config = Configuration::getDefaultConfiguration();
    $config->setAccessToken($accessToken);
    $config->setHost('https://api.xero.com');
    
    echo "2. Xero configuration set\n";
    
    // Create Guzzle client
    $guzzleClient = new Client([
        'verify' => false,
        'timeout' => 30
    ]);
    
    echo "3. Guzzle client created\n";
    
    // Create API instance
    $apiInstance = new AccountingApi($guzzleClient, $config);
    
    echo "4. API instance created\n";
    
    // Generate random data
    $invoiceData = [
        'jobId' => rand(1000, 9999),
        'customer' => 'Test Insurance Ltd',
        'customerAddress' => ['line1' => '123 Test Street', 'city' => 'London', 'postcode' => 'SW1A 1AA', 'country' => 'United Kingdom'],
        'customerPhone' => '0800 123 456',
        'customerEmail' => 'test@insurance.com',
        'description' => 'Test Wheel Service',
        'amount' => 0,
        'service' => 'Powder Coat',
        'regNo' => 'TEST123',
        'addons' => []
    ];
    
    echo "5. Test data generated\n";
    echo "   Customer: " . $invoiceData['customer'] . "\n";
    echo "   Service: " . $invoiceData['service'] . "\n\n";
    
    // Create contact
    $contact = new Contact();
    $contact->setName($invoiceData['customer']);
    
    $address = new Address();
    $address->setAddressType('STREET');
    $address->setAddressLine1($invoiceData['customerAddress']['line1']);
    $address->setCity($invoiceData['customerAddress']['city']);
    $address->setPostalCode($invoiceData['customerAddress']['postcode']);
    $address->setCountry($invoiceData['customerAddress']['country']);
    $contact->setAddresses([$address]);
    
    $contact->setPhones([
        (new Phone())
            ->setPhoneType('MOBILE')
            ->setPhoneNumber($invoiceData['customerPhone'])
    ]);
    
    $contact->setEmailAddress($invoiceData['customerEmail']);
    $contact->setIsCustomer(true);
    $contact->setContactStatus('ACTIVE');
    
    echo "6. Contact object created\n";
    
    // Create invoice
    $invoice = new Invoice();
    $invoice->setType('ACCREC');
    $invoice->setContact($contact);
    $invoice->setDate(new DateTime());
    $invoice->setDueDate(new DateTime('+30 days'));
    $invoice->setStatus('AUTHORISED');
    $invoice->setLineAmountTypes('Exclusive');
    
    echo "7. Invoice object created\n";
    
    // Create line item
    $lineItem = new LineItem();
    $lineItem->setDescription('Test Wheel Service - Powder Coating');
    $lineItem->setQuantity(1);
    $lineItem->setUnitAmount(250.00);
    $lineItem->setAccountCode('200');
    $lineItem->setTaxType('NONE');
    
    $invoice->setLineItems([$lineItem]);
    
    echo "8. Line item created\n";
    
    // Create invoices wrapper
    $invoices = new \XeroAPI\XeroPHP\Models\Accounting\Invoices();
    $invoices->setInvoices([$invoice]);
    
    echo "9. Invoices wrapper created\n";
    
    // Make API call
    echo "10. Making API call to Xero...\n";
    
    $result = $apiInstance->createInvoices($tenantId, $invoices);
    
    echo "11. API call successful!\n";
    echo "   Result type: " . get_class($result) . "\n";
    
    if ($result->getInvoices()) {
        $createdInvoice = $result->getInvoices()[0];
        echo "   Invoice ID: " . $createdInvoice->getInvoiceID() . "\n";
        echo "   Invoice Number: " . $createdInvoice->getInvoiceNumber() . "\n";
        echo "   Status: " . $createdInvoice->getStatus() . "\n";
        echo "   Total: " . $createdInvoice->getTotal() . "\n";
        echo "\n✅ SUCCESS: Invoice created in Xero!\n";
    } else {
        echo "❌ No invoices returned\n";
    }
    
} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
    echo "   File: " . $e->getFile() . "\n";
    echo "   Line: " . $e->getLine() . "\n";
    echo "   Trace: " . $e->getTraceAsString() . "\n";
}

echo "\n=== Debug Complete ===\n";
?>
