<?php
// Find recent invoices including the one we just created
echo "=== Finding Recent Invoices in Xero ===\n\n";

// Load stored tokens
$tokensFile = 'xero_tokens.json';
if (!file_exists($tokensFile)) {
    echo "❌ No tokens found. Please authorize the app first.\n";
    exit;
}

$tokens = json_decode(file_get_contents($tokensFile), true);
if (!$tokens || !isset($tokens['access_token'])) {
    echo "❌ Invalid token file. Please re-authorize the app.\n";
    exit;
}

$accessToken = $tokens['access_token'];
$tenantId = $tokens['tenant_id'];

echo "1. Fetching recent invoices from Xero...\n";

// Call Xero's invoices endpoint with date filter for recent invoices
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'https://api.xero.com/api.xro/2.0/Invoices');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Authorization: Bearer ' . $accessToken,
    'Accept: application/json',
    'Xero-tenant-id: ' . $tenantId
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

echo "2. API Response:\n";
echo "   HTTP Code: $httpCode\n";
if ($error) {
    echo "   Error: $error\n";
    exit;
}

// Parse the response
$data = json_decode($response, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    echo "❌ Failed to parse JSON response: " . json_last_error_msg() . "\n";
    exit;
}

if (isset($data['Invoices'])) {
    $invoices = $data['Invoices'];
    echo "   ✅ Found " . count($invoices) . " invoice(s) in Xero\n\n";
    
    echo "3. Recent invoices (last 10):\n";
    $recentInvoices = array_slice($invoices, 0, 10);
    
    foreach ($recentInvoices as $index => $invoice) {
        $invoiceNumber = $invoice['InvoiceNumber'] ?? 'Unknown';
        $contactName = $invoice['Contact']['Name'] ?? 'Unknown';
        $total = $invoice['Total'] ?? 'Unknown';
        $status = $invoice['Status'] ?? 'Unknown';
        $date = $invoice['Date'] ?? 'Unknown';
        
        // Check if this looks like our RefurbFlow invoice
        $isRefurbFlow = false;
        if (strpos($invoiceNumber, 'AUTO-') === 0 || 
            strpos($invoiceNumber, 'INV-') === 0 ||
            strpos($contactName, 'Test Customer') !== false ||
            strpos($contactName, 'More Than Insurance') !== false) {
            $isRefurbFlow = true;
        }
        
        $marker = $isRefurbFlow ? "🎯" : "  ";
        echo "   $marker Invoice #" . ($index + 1) . ":\n";
        echo "   ├─ Invoice Number: $invoiceNumber\n";
        echo "   ├─ Contact: $contactName\n";
        echo "   ├─ Total: $total\n";
        echo "   ├─ Status: $status\n";
        echo "   ├─ Date: $date\n";
        echo "   └─ Invoice ID: " . ($invoice['InvoiceID'] ?? 'Unknown') . "\n\n";
    }
    
    // Look specifically for our test invoice
    echo "4. Looking for our test invoice (INV-0046)...\n";
    $testInvoiceFound = false;
    foreach ($invoices as $invoice) {
        if (($invoice['InvoiceNumber'] ?? '') === 'INV-0046') {
            echo "   ✅ FOUND! Test invoice details:\n";
            echo "   ├─ Invoice ID: " . ($invoice['InvoiceID'] ?? 'Unknown') . "\n";
            echo "   ├─ Invoice Number: " . ($invoice['InvoiceNumber'] ?? 'Unknown') . "\n";
            echo "   ├─ Contact: " . ($invoice['Contact']['Name'] ?? 'Unknown') . "\n";
            echo "   ├─ Total: " . ($invoice['Total'] ?? 'Unknown') . "\n";
            echo "   ├─ Status: " . ($invoice['Status'] ?? 'Unknown') . "\n";
            echo "   ├─ Date: " . ($invoice['Date'] ?? 'Unknown') . "\n";
            echo "   └─ Due Date: " . ($invoice['DueDate'] ?? 'Unknown') . "\n";
            $testInvoiceFound = true;
            break;
        }
    }
    
    if (!$testInvoiceFound) {
        echo "   ❌ Test invoice INV-0046 not found in the list\n";
        echo "   This could mean:\n";
        echo "   - The invoice wasn't actually created\n";
        echo "   - It's in a different organization\n";
        echo "   - There's a delay in the API response\n";
    }
    
} else {
    echo "   ❌ Unexpected response format\n";
    echo "   Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
}

echo "\n=== Search Complete ===\n";
?>
