// In-House App JavaScript
$(document).ready(function() {
    // Initialize the app
    initializeApp();
    
    // Setup event listeners
    setupEventListeners();
    
    // Load initial data
    loadDashboardData();
    loadJobQueue();
    loadRecentActivity();
    
    // Initialize global search
    initializeGlobalSearch();
    
    // Initialize Xero integration
    initializeXeroIntegration();
});

function initializeApp() {
    // Check if staff is logged in
    const staff = localStorage.getItem('staff');
    if (!staff) {
        // Redirect to login (for now, just show demo data)
        console.log('No staff logged in, showing demo data');
    }
    
    // Set active navigation
    $('.nav-item').click(function(e) {
        e.preventDefault();
        const page = $(this).data('page');
        showPage(page);
        
        // Update active state
        $('.nav-item').removeClass('active');
        $(this).addClass('active');
    });
}

function setupEventListeners() {
    // Job management
    $('#filter-jobs').click(function() {
        filterJobs();
    });
    
    // Modal controls
    $('#close-job-modal').click(function() {
        $('#job-details-modal').addClass('hidden');
    });
    
    // Add job button
    $('#add-job-btn').click(function() {
        // This would open a form to add a new job
        showNotification('Add job feature coming soon!', 'info');
    });
    
    // Create Job functionality
    $('#create-job-btn, #create-job-btn-dashboard').click(function() {
        console.log('Create job button clicked');
        console.log('Modal element exists:', $('#create-job-modal').length > 0);
        console.log('Modal has hidden class:', $('#create-job-modal').hasClass('hidden'));
        $('#create-job-modal').removeClass('hidden');
        console.log('Modal hidden class removed');
    });
    
    $('#close-create-job-modal, #cancel-create-job').click(function() {
        $('#create-job-modal').addClass('hidden');
        clearCreateJobForm();
    });
    
    $('#submit-create-job').click(function() {
        createNewJob();
    });
    
    // Photo preview functionality
    $('#job-photos').change(function() {
        handlePhotoPreview(this);
    });
    
    // Customer selection functionality
    $('#customer-select').change(function() {
        handleCustomerSelection();
    });
    
    $('#add-customer-btn').click(function() {
        showAddCustomerForm();
    });
    
    // Customer page functionality
    $('#add-customer-btn-page').click(function() {
        // This would open a customer creation modal
        showNotification('Add customer feature coming soon!', 'info');
    });
    
    $('#customer-search').on('input', function() {
        filterCustomers($(this).val());
    });
    
    // Edit Customer Modal functionality
    $('#close-edit-customer-modal, #cancel-edit-customer').click(function() {
        $('#edit-customer-modal').addClass('hidden');
    });
    
    $('#save-edit-customer').click(function() {
        saveCustomerChanges();
    });
    
    // Settings page functionality
    $('#general-settings-tab').click(function() {
        switchSettingsTab('general');
    });
    
    $('#services-settings-tab').click(function() {
        switchSettingsTab('services');
    });
    
    $('#xero-settings-tab').click(function() {
        switchSettingsTab('xero');
    });
    
    $('#save-settings').click(function() {
        saveSettings();
    });
    
    $('#reset-settings').click(function() {
        resetSettings();
    });
    
    // Xero settings functionality
    $('#connect-xero-btn').click(function() {
        $('#xero-config-section').removeClass('hidden');
        $(this).addClass('hidden');
    });
    
    $('#test-xero-connection').click(function() {
        testXeroConnection();
    });
    
    $('#save-xero-config').click(function() {
        saveXeroConfiguration();
    });
    
    $('#add-service-btn').click(function() {
        openAddServiceModal();
    });
    
    // Add Service Modal functionality
    $('#close-add-service-modal, #cancel-add-service').click(function() {
        $('#add-service-modal').addClass('hidden');
    });
    
    $('#save-add-service').click(function() {
        saveNewService();
    });
    
    // Live preview updates
    $('#service-name, #service-description, #service-icon, #service-icon-color, #service-price, #service-price-type').on('input change', function() {
        updateServicePreview();
    });
    
    // Edit Customer Prices Modal functionality
    $('#close-edit-prices-modal, #cancel-edit-prices').click(function() {
        $('#edit-customer-prices-modal').addClass('hidden');
    });
    
    $('#save-edit-prices').click(function() {
        saveCustomerPrices();
    });
    
    // Toggle use default checkboxes
    $('#use-default-powder-coating').change(function() {
        togglePriceInput('powder-coating', this.checked);
    });
    
    $('#use-default-diamond-cutting').change(function() {
        togglePriceInput('diamond-cutting', this.checked);
    });
    
    $('#use-default-other-services').change(function() {
        togglePriceInput('other-services', this.checked);
    });
    
    // Update pricing summary on input change
    $('#edit-powder-coating-price, #edit-diamond-cutting-price, #edit-other-services-price').on('input', function() {
        updatePricingSummary();
    });
    
    // Logout button
    $('#logout-btn').click(function() {
        logout();
    });
    
    // Mobile menu toggle
    $('#mobile-menu-btn').click(function() {
        $('#sidebar').toggleClass('-translate-x-full');
        $('#mobile-overlay').toggleClass('hidden');
    });
    
    // Close mobile menu when clicking overlay
    $('#mobile-overlay').click(function() {
        $('#sidebar').addClass('-translate-x-full');
        $('#mobile-overlay').addClass('hidden');
    });
    
    // Close mobile menu when clicking nav items
    $('.nav-item').click(function() {
        if (window.innerWidth < 1024) { // lg breakpoint
            $('#sidebar').addClass('-translate-x-full');
            $('#mobile-overlay').addClass('hidden');
        }
    });
    
    // Handle window resize
    $(window).resize(function() {
        if (window.innerWidth >= 1024) { // lg breakpoint
            $('#sidebar').removeClass('-translate-x-full');
            $('#mobile-overlay').addClass('hidden');
        } else {
            $('#sidebar').addClass('-translate-x-full');
        }
    });
}

function showPage(pageName) {
    // Hide all pages
    $('.page-content').addClass('hidden');
    
    // Show selected page
    $(`#${pageName}-page`).removeClass('hidden');
    
    // Update navigation
    $('.nav-item').removeClass('active');
    $(`.nav-item[data-page="${pageName}"]`).addClass('active');
    
    // Load page-specific data
    switch(pageName) {
        case 'dashboard':
            loadDashboardData();
            loadJobQueue();
            loadRecentActivity();
            break;
        case 'jobs':
            loadJobsTable();
            break;
        case 'customers':
            loadCustomersTable();
            break;
        case 'settings':
            loadSettings();
            break;
        case 'activities':
            loadActivities();
            break;
        case 'invoices':
            loadInvoicesTable();
            break;
        case 'reports':
            loadReportsPage();
            break;
    }
}

function loadDashboardData() {
    $.ajax({
        url: 'data/dashboard.json',
        method: 'GET',
        dataType: 'json',
        success: function(data) {
            updateDashboardMetrics(data);
        },
        error: function() {
            // Fallback to demo data
            const demoData = {
                totalJobs: 24,
                inProgressJobs: 8,
                completedToday: 3,
                monthlyRevenue: 12450
            };
            updateDashboardMetrics(demoData);
        }
    });
}

function updateDashboardMetrics(data) {
    $('#total-jobs').text(data.totalJobs);
    $('#in-progress-jobs').text(data.inProgressJobs);
    $('#completed-today').text(data.completedToday);
    $('#monthly-revenue').text('£' + data.monthlyRevenue.toLocaleString());
}

// Store all jobs for search functionality
let allJobs = [];
let allCustomers = [];
let allInvoices = [];

// Global search data
let globalSearchData = {
    jobs: [],
    customers: [],
    invoices: []
};

// Xero Integration
let xeroApi = null;

// Xero Invoice Generation Functions
function initializeXeroIntegration() {
    // Check if XeroAPI is available
    if (typeof window.XeroAPI === 'undefined') {
        console.warn('Xero API helper not loaded. Invoice generation will be simulated.');
        return false;
    }
    
    try {
        console.log('Xero integration initialized successfully');
        return true;
    } catch (error) {
        console.error('Failed to initialize Xero integration:', error);
        return false;
    }
}

async function createXeroInvoice(job) {
    try {
        // Prepare invoice data
        const invoiceData = {
            jobId: job.id,
            customer: job.customer,
            amount: job.quote || 0,
            service: job.service || 'Wheel Refurbishment',
            description: job.description,
            regNo: job.regNo
        };
        
        // Use XeroAPI to create invoice
        const response = await window.XeroAPI.createInvoice(invoiceData);
        
        // Convert Xero response to our format
        const invoiceResult = {
            invoiceId: response.InvoiceID,
            invoiceNumber: response.InvoiceNumber,
            customer: response.Contact.Name,
            amount: invoiceData.amount,
            service: invoiceData.service,
            description: invoiceData.description,
            status: response.Status,
            createdDate: response.Date,
            dueDate: response.DueDate
        };
        
        return invoiceResult;
    } catch (error) {
        console.error('Failed to create Xero invoice:', error);
        throw error;
    }
}

function showXeroIntegrationModal() {
    const modalHtml = `
        <div id="xero-integration-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div class="bg-white rounded-lg p-6 max-w-md w-full mx-4">
                <div class="flex items-center justify-between mb-4">
                    <h3 class="text-lg font-semibold text-gray-900">Xero Integration Setup</h3>
                    <button onclick="closeXeroIntegrationModal()" class="text-gray-400 hover:text-gray-600">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <div class="space-y-4">
                    <div class="p-4 bg-blue-50 rounded-lg">
                        <div class="flex items-center">
                            <i class="fas fa-info-circle text-blue-600 mr-2"></i>
                            <span class="text-sm text-blue-800">Demo Mode Active</span>
                        </div>
                        <p class="text-xs text-blue-600 mt-1">
                            Invoices will be simulated for demonstration purposes.
                        </p>
                    </div>
                    
                    <div class="space-y-3">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Client ID</label>
                            <input type="text" id="xero-client-id" 
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Enter your Xero Client ID">
                        </div>
                        
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Client Secret</label>
                            <input type="password" id="xero-client-secret" 
                                   class="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                   placeholder="Enter your Xero Client Secret">
                        </div>
                    </div>
                    
                    <div class="flex space-x-3">
                        <button onclick="closeXeroIntegrationModal()" 
                                class="flex-1 px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                            Cancel
                        </button>
                        <button onclick="saveXeroCredentials()" 
                                class="flex-1 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700">
                            Save & Connect
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    $('body').append(modalHtml);
}

function closeXeroIntegrationModal() {
    $('#xero-integration-modal').remove();
}

// Xero Settings Functions
function loadXeroSettings() {
    // Update Xero status indicator
    updateXeroStatus();
    
    // Load saved configuration if available
    const savedConfig = localStorage.getItem('xero-config');
    if (savedConfig) {
        const config = JSON.parse(savedConfig);
        $('#xero-client-id').val(config.clientId || '');
        $('#xero-client-secret').val(config.clientSecret || '');
        $('#xero-redirect-uri').val(config.redirectUri || 'http://localhost:8080/in-house-app/index.html');
    }
}

function updateXeroStatus() {
    const isConnected = window.XeroConfig && !window.XeroConfig.demoMode && 
                       window.XeroConfig.clientId !== 'YOUR_XERO_CLIENT_ID';
    
    const statusElement = $('#xero-status');
    if (isConnected) {
        statusElement.html(`
            <div class="w-3 h-3 bg-green-500 rounded-full mr-2"></div>
            <span class="text-sm text-gray-600">Connected</span>
        `);
    } else {
        statusElement.html(`
            <div class="w-3 h-3 bg-red-500 rounded-full mr-2"></div>
            <span class="text-sm text-gray-600">Not Connected</span>
        `);
    }
}

async function testXeroConnection() {
    const clientId = $('#xero-client-id').val();
    const clientSecret = $('#xero-client-secret').val();
    
    if (!clientId || !clientSecret) {
        showNotification('Please enter both Client ID and Client Secret', 'error');
        return;
    }
    
    showNotification('Testing Xero connection...', 'info');
    
    try {
        const result = await window.XeroAPI.testConnection(clientId, clientSecret);
        showNotification('Xero connection test successful!', 'success');
        updateXeroStatus();
    } catch (error) {
        showNotification('Xero connection test failed: ' + error.message, 'error');
    }
}

function saveXeroConfiguration() {
    const clientId = $('#xero-client-id').val();
    const clientSecret = $('#xero-client-secret').val();
    const redirectUri = $('#xero-redirect-uri').val();
    
    if (!clientId || !clientSecret) {
        showNotification('Please enter both Client ID and Client Secret', 'error');
        return;
    }
    
    // Save configuration
    const config = {
        clientId: clientId,
        clientSecret: clientSecret,
        redirectUri: redirectUri,
        demoMode: false
    };
    
    localStorage.setItem('xero-config', JSON.stringify(config));
    
    // Update global config
    window.XeroConfig.clientId = clientId;
    window.XeroConfig.clientSecret = clientSecret;
    window.XeroConfig.redirectUri = redirectUri;
    window.XeroConfig.demoMode = false;
    
    // Initialize Xero integration
    if (initializeXeroIntegration()) {
        showNotification('Xero configuration saved successfully!', 'success');
        updateXeroStatus();
        $('#connect-xero-btn').addClass('hidden');
    } else {
        showNotification('Failed to initialize Xero integration', 'error');
    }
}

function saveXeroCredentials() {
    const clientId = $('#xero-client-id').val();
    const clientSecret = $('#xero-client-secret').val();
    
    if (!clientId || !clientSecret) {
        showNotification('Please enter both Client ID and Client Secret', 'error');
        return;
    }
    
    // Update config
    window.XeroConfig.clientId = clientId;
    window.XeroConfig.clientSecret = clientSecret;
    window.XeroConfig.demoMode = false;
    
    // Initialize Xero integration
    if (initializeXeroIntegration()) {
        showNotification('Xero integration configured successfully!', 'success');
        closeXeroIntegrationModal();
    } else {
        showNotification('Failed to initialize Xero integration', 'error');
    }
}

function loadJobQueue() {
    $.ajax({
        url: 'data/jobs.json?v=' + Date.now(),
        method: 'GET',
        dataType: 'json',
        success: function(data) {
            allJobs = data.jobs;
            displayJobQueue(allJobs);
            setupJobQueueSearch();
        },
        error: function() {
            // Fallback to demo data
            const demoJobs = [
                {
                    id: 1,
                    customer: "Direct Line Insurance",
                    description: "BMW 18\" Alloy Wheel Refurbishment",
                    service: "Powder Coating",
                    status: "in-progress",
                    priority: "high",
                    dueDate: "2024-01-20",
                    regNo: "BM18 ABC"
                },
                {
                    id: 2,
                    customer: "Aviva Insurance",
                    description: "Mercedes 19\" Diamond Cut Wheels",
                    service: "Diamond Cutting",
                    status: "pending",
                    priority: "medium",
                    dueDate: "2024-01-22",
                    regNo: "ME19 XYZ"
                },
                {
                    id: 3,
                    customer: "Admiral Insurance",
                    description: "Audi 17\" Standard Refurbishment",
                    service: "Other Services",
                    status: "quoted",
                    priority: "low",
                    dueDate: "2024-01-25",
                    regNo: "AU17 DEF"
                }
            ];
            allJobs = demoJobs;
            displayJobQueue(allJobs);
            setupJobQueueSearch();
        }
    });
}

function setupJobQueueSearch() {
    $('#job-queue-search').on('input', function() {
        const searchTerm = $(this).val().toLowerCase().trim();
        
        if (searchTerm === '') {
            // Show all jobs if search is empty
            displayJobQueue(allJobs);
        } else {
            // Filter jobs based on search term
            const filteredJobs = allJobs.filter(job => 
                job.customer.toLowerCase().includes(searchTerm) ||
                job.description.toLowerCase().includes(searchTerm) ||
                (job.regNo && job.regNo.toLowerCase().includes(searchTerm)) ||
                (job.service && job.service.toLowerCase().includes(searchTerm))
            );
            displayJobQueue(filteredJobs);
        }
    });
}

function displayJobQueue(jobs) {
    const queueContainer = $('#job-queue');
    queueContainer.empty();
    
    // Filter jobs to only show pending and in-progress (exclude ready, completed, invoiced, cancelled)
    const filteredJobs = jobs.filter(job => 
        job.status === 'pending' || job.status === 'in-progress'
    );
    
    // Sort jobs by status (pending first, then in-progress) then by priority and due date
    const sortedJobs = filteredJobs.sort((a, b) => {
        const statusOrder = { 
            'pending': 1, 
            'in-progress': 2
        };
        
        // First sort by status
        if (statusOrder[a.status] !== statusOrder[b.status]) {
            return statusOrder[a.status] - statusOrder[b.status];
        }
        
        // Then by priority
        const priorityOrder = { high: 3, medium: 2, low: 1 };
        if (priorityOrder[a.priority] !== priorityOrder[b.priority]) {
            return priorityOrder[b.priority] - priorityOrder[a.priority];
        }
        
        // Finally by due date
        return new Date(a.dueDate) - new Date(b.dueDate);
    });
    
    // Limit in-progress jobs to 3 maximum
    let pendingJobs = sortedJobs.filter(job => job.status === 'pending');
    let inProgressJobs = sortedJobs.filter(job => job.status === 'in-progress').slice(0, 3);
    
    // Combine and display
    const displayJobs = [...pendingJobs, ...inProgressJobs];
    
    displayJobs.forEach(job => {
        const jobElement = createJobQueueItem(job);
        queueContainer.append(jobElement);
    });
}

function createJobQueueItem(job) {
    const statusClass = getStatusClass(job.status);
    const statusText = getStatusText(job.status);
    const statusIcon = getStatusIcon(job.status);
    const priorityClass = getPriorityClass(job.priority);
    const priorityColor = getPriorityColor(job.priority);
    
    return $(`
        <div class="job-queue-card group bg-white border border-gray-100 rounded-2xl p-5 hover:shadow-xl hover:border-[#00a9ea]/30 transition-all duration-300 cursor-pointer" onclick="showJobDetails(${job.id})">
            <!-- Header Section -->
            <div class="flex items-start justify-between mb-4">
                <div class="flex items-center space-x-3">
                    <div class="w-12 h-12 bg-gradient-to-br from-[#00a9ea]/10 to-[#00a9ea]/20 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform duration-200">
                        <svg width="30px" height="30px" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" stroke="#5cc0ff"><g id="SVGRepo_bgCarrier" stroke-width="0"></g><g id="SVGRepo_tracerCarrier" stroke-linecap="round" stroke-linejoin="round"></g><g id="SVGRepo_iconCarrier"> <circle cx="12" cy="12" r="2" stroke="#2694d9" stroke-width="1.5"></circle> <path d="M6 12L10 12" stroke="#2694d9" stroke-width="1.5" stroke-linecap="round"></path> <path d="M14 12L18 12" stroke="#2694d9" stroke-width="1.5" stroke-linecap="round"></path> <path d="M9 17.1963L11 13.7322" stroke="#2694d9" stroke-width="1.5" stroke-linecap="round"></path> <path d="M13 10.2681L15 6.80396" stroke="#2694d9" stroke-width="1.5" stroke-linecap="round"></path> <path d="M15 17.1963L13 13.7322" stroke="#2694d9" stroke-width="1.5" stroke-linecap="round"></path> <path d="M11 10.2681L9 6.80396" stroke="#2694d9" stroke-width="1.5" stroke-linecap="round"></path> <path d="M7 3.33782C8.47087 2.48697 10.1786 2 12 2C17.5228 2 22 6.47715 22 12C22 17.5228 17.5228 22 12 22C6.47715 22 2 17.5228 2 12C2 10.1786 2.48697 8.47087 3.33782 7" stroke="#2694d9" stroke-width="1.5" stroke-linecap="round"></path> <path d="M15 17.1973C14.1175 17.7078 13.0929 18 12 18C8.68629 18 6 15.3137 6 12C6 8.68629 8.68629 6 12 6C15.3137 6 18 8.68629 18 12C18 13.0929 17.7078 14.1175 17.1973 15" stroke="#2694d9" stroke-width="1.5" stroke-linecap="round"></path> </g></svg>
                        </div>
                        <div>
                        <h3 class="font-semibold text-gray-900 text-lg leading-tight">${job.description || 'Job Description'}</h3>
                        <div class="flex items-center space-x-2 mt-1">
                            <span class="text-sm text-gray-500">#${job.id}</span>
                            <span class="w-1 h-1 bg-gray-300 rounded-full"></span>
                            <div class="flex items-center space-x-1.5">
                                <i class="fas fa-building text-gray-400 text-xs"></i>
                                <span class="text-sm text-gray-600">${job.customer}</span>
                        </div>
                    </div>
                    </div>
                </div>
                        <div class="flex items-center space-x-2">
                    <span class="status-badge-modern ${statusClass} flex items-center px-3 py-1.5 rounded-full text-xs font-medium">
                        <i class="${statusIcon} mr-1.5"></i>
                                ${statusText}
                            </span>
                    <span class="priority-badge-modern ${priorityClass} flex items-center px-2.5 py-1 rounded-full text-xs font-medium" style="background-color: ${priorityColor}; color: white;">
                        <i class="fas fa-flag mr-1"></i>
                        ${job.priority.toUpperCase()}
                    </span>
                        </div>
                        </div>

            <!-- Content Section -->
            <div class="space-y-3">
                <!-- Job Description with Service Type -->
                <div class="flex items-center space-x-3">
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium ${job.service === 'Diamond Cut' ? 'bg-purple-100 text-purple-800' : 'bg-blue-100 text-blue-800'}">
                        ${job.service || 'Service Not Specified'}
                    </span>
                    <span class="text-gray-700 font-medium">${job.description}</span>
                    </div>
                
                <!-- Additional Info Row -->
                <div class="flex items-center justify-between">
                    <div class="flex items-center space-x-4">
                        ${job.regNo ? `
                            <div class="flex items-center space-x-2">
                                <div class="w-6 h-6 bg-yellow-50 rounded-lg flex items-center justify-center">
                                    <i class="fas fa-car text-yellow-600 text-xs"></i>
                </div>
                                <span class="reg-plate reg-plate-small">${job.regNo}</span>
                            </div>
                        ` : ''}
                        ${job.notes && job.notes.length > 0 ? `
                            <div class="flex items-center space-x-2">
                                <div class="w-6 h-6 bg-amber-50 rounded-lg flex items-center justify-center">
                                    <i class="fas fa-sticky-note text-amber-600 text-xs"></i>
                                </div>
                                <span class="text-sm text-gray-600">${job.notes.length} note${job.notes.length > 1 ? 's' : ''}</span>
                            </div>
                        ` : ''}
                    </div>
                </div>

                <!-- Due Date & Actions -->
                <div class="flex items-center justify-between pt-2 border-t border-gray-50">
                    <div class="flex items-center space-x-2 text-sm text-gray-500">
                        <i class="fas fa-calendar-alt"></i>
                        <span>Due: ${formatDate(job.dueDate)}</span>
                    </div>
                    <div class="flex items-center space-x-2">
                        ${job.status === 'in-progress' ? 
                            `<button class="action-btn-modern ready-btn px-3 py-1.5 bg-green-50 text-green-700 hover:bg-green-100 rounded-lg text-xs font-medium transition-colors" data-job="${job.id}" title="Mark as Ready" onclick="event.stopPropagation()">
                                <i class="fas fa-check-circle mr-1"></i>Ready
                            </button>` : ''
                        }
                        ${job.status === 'ready' ? 
                            `<button class="action-btn-modern invoice-btn px-3 py-1.5 bg-purple-50 text-purple-700 hover:bg-purple-100 rounded-lg text-xs font-medium transition-colors" data-job="${job.id}" title="Mark as Invoiced" onclick="event.stopPropagation()">
                                <i class="fas fa-file-invoice mr-1"></i>Invoiced
                            </button>` : ''
                        }
                        <button class="action-btn-modern view-btn px-3 py-1.5 bg-gray-50 text-gray-700 hover:bg-gray-100 rounded-lg text-xs font-medium transition-colors" onclick="event.stopPropagation(); showJobDetails(${job.id})" title="View Details">
                            <i class="fas fa-eye mr-1"></i>View
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `);
}

function loadJobsTable() {
    console.log('Loading jobs table...');
    $.ajax({
        url: 'data/jobs.json?v=' + Date.now(),
        method: 'GET',
        dataType: 'json',
        success: function(data) {
            console.log('Jobs loaded successfully:', data.jobs.length, 'jobs');
            console.log('First few jobs:', data.jobs.slice(0, 3));
            displayJobsTable(data.jobs);
        },
        error: function(xhr, status, error) {
            console.log('Error loading jobs:', error);
            // Fallback to demo data
            const demoJobs = [
                {
                    id: 1,
                    customer: "Direct Line Insurance",
                    description: "BMW 18\" Alloy Wheel Refurbishment",
                    regNo: "AB12 CDE",
                    status: "in-progress",
                    dueDate: "2024-01-20"
                },
                {
                    id: 2,
                    customer: "Aviva Insurance",
                    description: "Mercedes 19\" Diamond Cut Wheels",
                    regNo: "XY98 ZAB",
                    status: "pending",
                    dueDate: "2024-01-22"
                },
                {
                    id: 3,
                    customer: "Admiral Insurance",
                    description: "Audi 17\" Standard Refurbishment",
                    regNo: "MN34 PQR",
                    status: "invoiced",
                    dueDate: "2024-01-25"
                }
            ];
            console.log('Using demo data:', demoJobs.length, 'jobs');
            displayJobsTable(demoJobs);
        }
    });
}

function displayJobsTable(jobs) {
    const tableBody = $('#jobs-table-body');
    console.log('Table body element:', tableBody.length);
    tableBody.empty();
    
    console.log('Displaying jobs:', jobs.length, 'jobs');
    
    // Sort jobs by status: pending, in-progress, invoiced, ready, completed
    const sortedJobs = sortJobsByStatus(jobs);
    console.log('Jobs sorted by status:', sortedJobs.map(job => ({ id: job.id, status: job.status })));
    
    sortedJobs.forEach((job, index) => {
        console.log(`Processing job ${index + 1}:`, job.id, job.customer, job.status);
        const row = createJobTableRow(job);
        tableBody.append(row);
    });
    
    console.log('Total rows in table:', tableBody.find('tr').length);
}

function sortJobsByStatus(jobs) {
    const statusOrder = {
        'pending': 1,
        'in-progress': 2,
        'invoiced': 3,
        'ready': 4,
        'completed': 5,
        'cancelled': 6
    };
    
    return jobs.sort((a, b) => {
        const statusA = statusOrder[a.status] || 999;
        const statusB = statusOrder[b.status] || 999;
        
        // If same status, sort by due date (earliest first)
        if (statusA === statusB) {
            return new Date(a.dueDate) - new Date(b.dueDate);
        }
        
        return statusA - statusB;
    });
}

function createJobTableRow(job) {
    console.log('Creating row for job:', job.id, job.customer);
    const statusClass = getStatusClass(job.status);
    const statusText = getStatusText(job.status);
    
    return $(`
        <tr class="table-row">
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">#${job.id}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${job.customer}</td>
            <td class="px-6 py-4 whitespace-nowrap">
                ${job.regNo ? `<span class="reg-plate reg-plate-small">${job.regNo}</span>` : '<span class="text-gray-400 text-sm">-</span>'}
            </td>
            <td class="px-6 py-4 text-sm text-gray-500">${job.description}</td>
            <td class="px-6 py-4 whitespace-nowrap">
                <span class="status-badge ${statusClass}">${statusText}</span>
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                ${job.notes && job.notes.length > 0 ? 
                    `<span class="note-badge">
                        <i class="fas fa-sticky-note note-icon"></i>
                        ${job.notes.length}
                    </span>` : 
                    '<span class="text-gray-400 text-sm">-</span>'
                }
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${formatDate(job.dueDate)}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                <div class="flex items-center space-x-2">
                    ${job.status === 'in-progress' ? 
                        `<button class="action-btn ready-btn" data-job="${job.id}" title="Mark as Ready">
                            <i class="fas fa-check-circle mr-1"></i>Ready
                        </button>` : ''
                    }
                    ${job.status === 'ready' ? 
                        `<button class="action-btn invoice-btn" data-job="${job.id}" title="Mark as Invoiced">
                            <i class="fas fa-file-invoice mr-1"></i>Invoiced
                        </button>` : ''
                    }
                    <button onclick="showJobDetails(${job.id})" class="text-blue-600 hover:text-blue-900">View</button>
                    <button onclick="editJob(${job.id})" class="text-green-600 hover:text-green-900">Edit</button>
                </div>
            </td>
        </tr>
    `);
}

function loadRecentActivity() {
    const activities = [
        { time: "10:30 AM", action: "Job #5 completed", user: "John Smith", icon: "fas fa-check-circle", color: "text-green-500" },
        { time: "09:45 AM", action: "Quote sent for Job #8", user: "Sarah Johnson", icon: "fas fa-file-invoice", color: "text-[#00a9ea]" },
        { time: "09:15 AM", action: "New job #9 received", user: "System", icon: "fas fa-plus", color: "text-blue-500" },
        { time: "08:30 AM", action: "Job #3 started", user: "Mike Wilson", icon: "fas fa-play", color: "text-amber-500" }
    ];
    
    const activityContainer = $('#recent-activity');
    activityContainer.empty();
    
    activities.forEach(activity => {
        const activityElement = $(`
            <div class="flex items-center space-x-4 py-3 border-b border-gray-100 last:border-b-0">
                <div class="w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center">
                    <i class="${activity.icon} ${activity.color} text-sm"></i>
                </div>
                <div class="flex-1">
                    <p class="text-sm text-gray-800 font-medium">${activity.action}</p>
                    <p class="text-xs text-gray-500">by ${activity.user}</p>
                </div>
                <span class="text-xs text-gray-400 font-medium">${activity.time}</span>
            </div>
        `);
        activityContainer.append(activityElement);
    });
}

function showJobDetails(jobId) {
    // This would load job details from the server
    const jobDetails = {
        id: jobId,
        customer: "Direct Line Insurance",
        phone: "+44 20 7123 4567",
        email: "claims@directline.com",
        description: "BMW 18\" Alloy Wheel Refurbishment",
        regNo: "AB12 CDE",
        status: "in-progress",
        priority: "high",
        dueDate: "2024-01-20",
        completionDate: "2024-01-18",
        quantity: 4,
        quote: 320.00,
        notes: [
            {
                id: 1,
                text: "Customer requested diamond cut finish",
                author: "John Smith",
                timestamp: "2024-01-15T09:30:00Z",
                type: "general"
            },
            {
                id: 2,
                text: "Wheels have minor curb damage on front left",
                author: "John Smith",
                timestamp: "2024-01-15T10:15:00Z",
                type: "damage"
            }
        ],
        invoices: [
            {
                invoiceNumber: "INV-2024-001",
                amount: 320.00,
                issuedDate: "2024-01-18",
                dueDate: "2024-02-17",
                status: "pending",
                paymentMethod: "Bank Transfer"
            }
        ],
        photos: ["../shared-resources/images/rim1.webp", "../shared-resources/images/rim2.jpg"],
        addons: [
            {
                id: 1,
                type: "rush-service",
                description: "24-hour completion guarantee",
                quantity: 1,
                price: 50.00,
                addedAt: "2024-01-15T10:30:00Z"
            },
            {
                id: 2,
                type: "premium-finish",
                description: "High-gloss clear coat finish",
                quantity: 4,
                price: 25.00,
                addedAt: "2024-01-15T11:15:00Z"
            }
        ]
    };
    
    const detailsContent = $(`
        <div class="space-y-6">
            <!-- Tabs Navigation -->
            <div class="job-tabs">
                <div class="flex space-x-0">
                    <div class="job-tab active" data-tab="overview">
                        <i class="fas fa-info-circle mr-2"></i>Overview
                    </div>
                    <div class="job-tab" data-tab="notes">
                        <i class="fas fa-sticky-note mr-2"></i>Notes
                        ${jobDetails.notes && jobDetails.notes.length > 0 ? `<span class="note-badge ml-2">${jobDetails.notes.length}</span>` : ''}
                    </div>
                    <div class="job-tab" data-tab="addons">
                        <i class="fas fa-plus-circle mr-2"></i>Add-ons
                        ${jobDetails.addons && jobDetails.addons.length > 0 ? `<span class="note-badge ml-2">${jobDetails.addons.length}</span>` : ''}
                    </div>
                    <div class="job-tab" data-tab="invoices">
                        <i class="fas fa-file-invoice mr-2"></i>Invoices
                        ${jobDetails.invoices && jobDetails.invoices.length > 0 ? `<span class="note-badge ml-2">${jobDetails.invoices.length}</span>` : ''}
                    </div>
                </div>
            </div>


            <!-- Overview Tab Content -->
            <div class="job-tab-content active" id="overview-tab">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                    <h3 class="text-lg font-semibold mb-4">Job Information</h3>
                    <div class="space-y-3">
                        <div>
                            <label class="font-medium text-gray-700">Job ID:</label>
                            <p class="text-gray-600">#${jobDetails.id}</p>
                        </div>
                        <div>
                            <label class="font-medium text-gray-700">Description:</label>
                            <p class="text-gray-600">${jobDetails.description}</p>
                        </div>
                        <div>
                            <label class="font-medium text-gray-700">Status:</label>
                            <span class="status-badge ${getStatusClass(jobDetails.status)}">${getStatusText(jobDetails.status)}</span>
                        </div>
                        <div>
                            <label class="font-medium text-gray-700">Due Date:</label>
                            <p class="text-gray-600">${formatDate(jobDetails.dueDate)}</p>
                        </div>
                        <div>
                            <label class="font-medium text-gray-700">Quote:</label>
                            <p class="text-gray-600">£${jobDetails.quote}</p>
                        </div>
                            ${jobDetails.completionDate ? `
                            <div>
                                <label class="font-medium text-gray-700">Completed Date:</label>
                                <p class="text-gray-600">${formatDate(jobDetails.completionDate)}</p>
                            </div>
                            ` : ''}
                    </div>
                </div>
                <div>
                    <h3 class="text-lg font-semibold mb-4">Photos</h3>
                    <div class="photo-gallery">
                        ${jobDetails.photos.map(photo => `
                            <div class="photo-item">
                                    <img src="${photo}" alt="Job photo" class="photo-image" data-photo="${photo}" title="Double-tap to enlarge">
                                <div class="photo-overlay">
                                        <div class="flex space-x-2">
                                            <button class="photo-btn enlarge-btn" data-photo="${photo}" title="Enlarge Photo">
                                                <i class="fas fa-search-plus"></i>
                                            </button>
                                            <button class="photo-btn download-btn" data-photo="${photo}" data-filename="${photo.split('/').pop()}" title="Download Photo">
                                                <i class="fas fa-download"></i>
                                            </button>
                                        </div>
                                </div>
                            </div>
                        `).join('')}
                    </div>
                </div>
            </div>
            </div>

            <!-- Notes Tab Content -->
            <div class="job-tab-content" id="notes-tab">
                <div class="space-y-4">
                    <div class="flex justify-between items-center">
                        <h3 class="text-lg font-semibold">Job Notes</h3>
                        <button id="add-note-btn" class="btn-modern btn-modern-primary font-bold">
                            <i class="fas fa-plus mr-2"></i>Add Note
                        </button>
                    </div>
                    <div id="add-note-form" class="mt-4 hidden">
                        <div class="bg-gray-50 rounded-xl p-4">
                            <textarea id="new-note-text" placeholder="Enter your note here..." class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00a9ea] focus:border-[#00a9ea] resize-none" rows="3"></textarea>
                            <div class="flex justify-end space-x-3 mt-3">
                                <button id="cancel-note-btn" class="px-4 py-2 text-gray-600 hover:text-gray-800 transition-colors">
                                    Cancel
                                </button>
                                <button id="save-note-btn" class="px-6 py-2 bg-[#00a9ea] text-white rounded-lg hover:bg-[#0088cc] transition-colors font-medium">
                                    <i class="fas fa-save mr-2"></i>Save Note
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="notes-container">
                        ${jobDetails.notes && jobDetails.notes.length > 0 ? 
                            jobDetails.notes.map(note => `
                                <div class="note-item note-${note.type}">
                                    <div class="note-text">${note.text}</div>
                                    <div class="note-meta">
                                        <span class="note-author">${note.author}</span>
                                        <span class="note-timestamp">${formatDateTime(note.timestamp)}</span>
                                    </div>
                                </div>
                            `).join('') : 
                            '<div class="text-center py-8 text-gray-500"><i class="fas fa-sticky-note text-4xl mb-4"></i><p>No notes yet</p></div>'
                        }
                    </div>
                </div>
            </div>

            <!-- Add-ons Tab Content -->
            <div class="job-tab-content" id="addons-tab">
                <div class="space-y-4">
                    <div class="flex justify-between items-center">
                        <h3 class="text-lg font-semibold">Add-ons</h3>
                        <button id="add-addon-btn" class="btn-modern btn-modern-primary font-bold">
                            <i class="fas fa-plus mr-2"></i>Add On
                        </button>
                    </div>
                    <div id="add-addon-form" class="hidden">
                        <div class="bg-gray-50 rounded-xl p-4">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Add-on Type</label>
                                    <select id="addon-type" class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00a9ea] focus:border-[#00a9ea]">
                                        <option value="">Select add-on type</option>
                                        <option value="extra-wheels">Extra Wheels</option>
                                        <option value="rush-service">Rush Service</option>
                                        <option value="premium-finish">Premium Finish</option>
                                        <option value="warranty-extension">Warranty Extension</option>
                                        <option value="pickup-delivery">Pickup & Delivery</option>
                                        <option value="other">Other</option>
                                    </select>
            </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Description</label>
                                    <input type="text" id="addon-description" placeholder="Add-on description" class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00a9ea] focus:border-[#00a9ea]">
            </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Quantity</label>
                                    <input type="number" id="addon-quantity" min="1" value="1" class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00a9ea] focus:border-[#00a9ea]">
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Price (£)</label>
                                    <input type="number" id="addon-price" step="0.01" min="0" placeholder="0.00" class="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#00a9ea] focus:border-[#00a9ea]">
                                </div>
                            </div>
                            <div class="flex justify-end space-x-3 mt-4">
                                <button id="cancel-addon-btn" class="px-4 py-2 text-gray-600 hover:text-gray-800 transition-colors">
                                    Cancel
                                </button>
                                <button id="save-addon-btn" class="px-6 py-2 bg-[#00a9ea] text-white rounded-lg hover:bg-[#0088cc] transition-colors font-medium">
                                    <i class="fas fa-save mr-2"></i>Add to Job
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="addons-container">
                        ${jobDetails.addons && jobDetails.addons.length > 0 ? 
                            jobDetails.addons.map(addon => `
                                <div class="addon-item">
                                    <div class="flex justify-between items-start">
                                        <div class="flex-1">
                                            <h4 class="font-medium text-gray-900">${addon.type}</h4>
                                            <p class="text-sm text-gray-600">${addon.description}</p>
                                            <div class="flex items-center space-x-4 mt-1">
                                                <span class="text-xs text-gray-500">Qty: ${addon.quantity}</span>
                                                <span class="text-sm font-medium text-gray-900">£${addon.price.toFixed(2)}</span>
                                            </div>
                                        </div>
                                        <button class="remove-addon-btn text-red-500 hover:text-red-700 transition-colors" data-addon-id="${addon.id}">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </div>
                            `).join('') : 
                            '<div class="text-center text-gray-500 py-8"><i class="fas fa-plus-circle text-4xl mb-2"></i><p>No add-ons yet</p></div>'
                        }
                    </div>
                </div>
            </div>

            <!-- Invoices Tab Content -->
            <div class="job-tab-content" id="invoices-tab">
                <div class="space-y-4">
                    <div class="flex justify-between items-center">
                        <h3 class="text-lg font-semibold">Invoices Issued</h3>
                        
                    </div>
                    <div class="invoices-container">
                        ${jobDetails.invoices && jobDetails.invoices.length > 0 ? 
                            jobDetails.invoices.map(invoice => `
                                <div class="bg-white border border-gray-200 rounded-xl p-6 hover:shadow-lg transition-all duration-200">
                                    <div class="flex justify-between items-start mb-4">
                                        <div>
                                            <h4 class="font-semibold text-gray-900">Invoice #${invoice.invoiceNumber}</h4>
                                            <p class="text-sm text-gray-500">Issued: ${formatDate(invoice.issuedDate)}</p>
                                        </div>
                                        <div class="text-right">
                                            <p class="text-lg font-semibold text-gray-900">£${invoice.amount}</p>
                                            <span class="status-badge ${invoice.status === 'paid' ? 'status-completed' : 'status-pending'}">${invoice.status === 'paid' ? 'Paid' : 'Pending'}</span>
                                        </div>
                                    </div>
                                    <div class="flex justify-between items-center">
                                        <div class="text-sm text-gray-600">
                                            <p>Due: ${formatDate(invoice.dueDate)}</p>
                                            <p>Payment Method: ${invoice.paymentMethod || 'Not specified'}</p>
                                        </div>
                                        <div class="flex space-x-2">
                                            <button class="px-3 py-1 bg-blue-100 text-blue-700 rounded-lg text-sm hover:bg-blue-200 transition-colors">
                                                <i class="fas fa-download mr-1"></i>Download
                                            </button>
                                            <button class="px-3 py-1 bg-green-100 text-green-700 rounded-lg text-sm hover:bg-green-200 transition-colors">
                                                <i class="fas fa-eye mr-1"></i>View
                                            </button>
                                            ${invoice.status === 'pending' ? `
                                            <button class="px-3 py-1 bg-emerald-100 text-emerald-700 rounded-lg text-sm hover:bg-emerald-200 transition-colors mark-paid-btn" 
                                                    data-invoice="${invoice.invoiceNumber}" 
                                                    data-job="${jobDetails.id}">
                                                <i class="fas fa-check mr-1"></i>Mark as Paid
                                            </button>
                                            ` : ''}
                                        </div>
                                    </div>
                                </div>
                            `).join('') : 
                            '<div class="text-center py-8 text-gray-500"><i class="fas fa-file-invoice text-4xl mb-4"></i><p>No invoices issued yet</p></div>'
                        }
                    </div>
                </div>
            </div>

        </div>
    `);
    
    // Store job details globally for access by other functions
    window.currentJobDetails = jobDetails;
    
    $('#job-details-content').html(detailsContent);
    
    // Update modal header with job information
    const headerContent = `
        <div class="flex items-center justify-between py-2 w-full">
            <div class="flex items-center space-x-12">
                <div>
                    <h2 class="text-3xl font-bold text-gray-900 mb-2">Job #${jobDetails.id}</h2>
                    <div class="flex items-center mb-2">
                        <i class="fas fa-building text-gray-500 mr-2"></i>
                        <p class="text-lg text-gray-600">${jobDetails.customer}</p>
                    </div>
                    <div class="flex items-center space-x-4 text-sm text-gray-500">
                        <div class="flex items-center">
                            <i class="fas fa-phone mr-1"></i>
                            <span>${jobDetails.phone || 'N/A'}</span>
                        </div>
                        <div class="flex items-center">
                            <i class="fas fa-envelope mr-1"></i>
                            <span>${jobDetails.email || 'N/A'}</span>
                        </div>
                    </div>
                </div>
                ${jobDetails.regNo ? `
                <div class="pl-8 border-l border-gray-200">
                    <p class="text-sm text-gray-500 mb-2">Registration</p>
                    <span class="reg-plate reg-plate-large">${jobDetails.regNo}</span>
                </div>
                ` : ''}
            </div>
            <div class="flex items-center space-x-8">
                <div class="text-right">
                    <p class="text-sm text-gray-500 mb-2">Status</p>
                    <span class="status-badge ${getStatusClass(jobDetails.status)} text-base px-4 py-2">
                        <i class="${getStatusIcon(jobDetails.status)} mr-2"></i>
                        ${getStatusText(jobDetails.status)}
                    </span>
                </div>
                <button id="close-job-modal" class="w-12 h-12 bg-gray-100 hover:bg-gray-200 rounded-xl flex items-center justify-center transition-colors">
                    <i class="fas fa-times text-gray-600 text-lg"></i>
                </button>
            </div>
        </div>
    `;
    
    // Update modal header
    $('#job-details-modal .modal-content > div:first-child').html(headerContent);
    
    // Add modal footer with action buttons
    const footerContent = `
        <div class="modal-footer">
            <div class="flex justify-between items-center">
                <div class="flex items-center space-x-3">
                ${jobDetails.status === 'in-progress' ? 
                    `<button class="action-btn ready-btn" data-job="${jobDetails.id}" title="Mark as Ready">
                        <i class="fas fa-check-circle mr-2"></i>Ready
                    </button>` : ''
                }
                ${jobDetails.status === 'ready' ? 
                    `<button class="action-btn invoice-btn" data-job="${jobDetails.id}" title="Mark as Invoiced">
                        <i class="fas fa-file-invoice mr-2"></i>Invoiced
                    </button>` : ''
                }
                </div>
                <div class="flex items-center space-x-4">
                    <button class="px-6 py-3 bg-gray-50 text-gray-600 font-medium rounded-xl border border-gray-200 hover:bg-gray-100 hover:border-gray-300 focus:outline-none focus:ring-2 focus:ring-gray-300 focus:ring-offset-2 transition-all duration-200 transform hover:scale-105" onclick="$('#job-details-modal').addClass('hidden')">
                        <i class="fas fa-times mr-2 text-sm"></i>Close
                    </button>
                    <button class="px-6 py-3 bg-gradient-to-r from-[#00a9ea] to-[#0088cc] text-white font-semibold rounded-xl shadow-sm hover:shadow-lg focus:outline-none focus:ring-2 focus:ring-[#00a9ea] focus:ring-offset-2 transition-all duration-200 transform hover:scale-105 hover:-translate-y-0.5" onclick="editJob(${jobDetails.id})">
                        <i class="fas fa-edit mr-2 text-sm"></i>Edit Job
                    </button>
                </div>
            </div>
        </div>
    `;
    
    // Remove existing footer if any
    $('#job-details-modal .modal-footer').remove();
    
    // Add footer after modal content
    $('#job-details-modal .modal-content').append(footerContent);
    
    $('#job-details-modal').removeClass('hidden');
    
    // Add tab switching functionality
    $('.job-tab').click(function() {
        const tabName = $(this).data('tab');
        
        // Remove active class from all tabs and content
        $('.job-tab').removeClass('active');
        $('.job-tab-content').removeClass('active');
        
        // Add active class to clicked tab and corresponding content
        $(this).addClass('active');
        $(`#${tabName}-tab`).addClass('active');
    });
    
    // Add event listener for mark paid buttons
    $('.mark-paid-btn').off('click').on('click', function() {
        const invoiceNumber = $(this).data('invoice');
        const jobId = $(this).data('job');
        console.log('Mark paid button clicked via event listener');
        markInvoiceAsPaid(invoiceNumber, jobId);
    });
    
    // Add event listeners for Ready and Invoice buttons
    $('.ready-btn').off('click').on('click', function() {
        const jobId = $(this).data('job');
        markJobAsReady(jobId);
    });
    
    $('.invoice-btn').off('click').on('click', function() {
        const jobId = $(this).data('job');
        markJobAsInvoiced(jobId);
    });
    
    // Photo functionality
    $('.enlarge-btn').off('click').on('click', function() {
        const photoSrc = $(this).data('photo');
        enlargePhoto(photoSrc);
    });
    
    $('.download-btn').off('click').on('click', function() {
        const photoSrc = $(this).data('photo');
        const filename = $(this).data('filename');
        downloadPhoto(photoSrc, filename);
    });
    
    // Double-tap to enlarge photos
    $('.photo-image').off('dblclick').on('dblclick', function() {
        const photoSrc = $(this).data('photo');
        enlargePhoto(photoSrc);
    });
    
    // Add Note functionality
    $('#add-note-btn').off('click').on('click', function() {
        $('#add-note-form').slideDown(300);
        $('#new-note-text').focus();
    });
    
    $('#cancel-note-btn').off('click').on('click', function() {
        $('#add-note-form').slideUp(300);
        $('#new-note-text').val('');
    });
    
    $('#save-note-btn').off('click').on('click', function() {
        const noteText = $('#new-note-text').val().trim();
        if (noteText) {
            addNewNote(noteText, jobDetails.id);
            $('#add-note-form').slideUp(300);
            $('#new-note-text').val('');
        }
    });
    
    // Add-on functionality
    $('#add-addon-btn').off('click').on('click', function() {
        $('#add-addon-form').slideDown(300);
        $('#addon-type').focus();
    });
    
    
    $('#cancel-addon-btn').off('click').on('click', function() {
        $('#add-addon-form').slideUp(300);
        clearAddonForm();
    });
    
    $('#save-addon-btn').off('click').on('click', function() {
        const addonData = {
            type: $('#addon-type').val(),
            description: $('#addon-description').val().trim(),
            quantity: parseInt($('#addon-quantity').val()) || 1,
            price: parseFloat($('#addon-price').val()) || 0
        };
        
        if (addonData.type && addonData.description) {
            addNewAddon(addonData, jobDetails.id);
            $('#add-addon-form').slideUp(300);
            clearAddonForm();
        } else {
            showNotification('Please fill in all required fields', 'error');
        }
    });
    
    // Remove add-on functionality
    $('.remove-addon-btn').off('click').on('click', function() {
        const addonId = $(this).data('addon-id');
        removeAddon(addonId, jobDetails.id);
    });
}

function formatDateTime(timestamp) {
    const date = new Date(timestamp);
    return date.toLocaleString('en-GB', {
        day: '2-digit',
        month: 'short',
        year: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

// Test function to verify onclick is working
function testButton() {
    console.log('Test button clicked!');
    showNotification('Test button works!', 'success');
}

function markJobAsReady(jobId) {
    console.log('Marking job as ready:', jobId);
    
    // Find the job data
    const job = allJobs.find(j => j.id == jobId);
    if (!job) {
        showNotification('Job not found', 'error');
        return;
    }
    
    // Show loading notification
    showNotification('Marking job as ready and creating invoice...', 'info');
    
    // Create invoice in Xero
    createXeroInvoice(job)
        .then(invoiceData => {
            // Update job status
            job.status = 'ready';
            
            // Update job status in the current job details if it matches
            if (window.currentJobDetails && window.currentJobDetails.id == jobId) {
                window.currentJobDetails.status = 'ready';
                
                // Update the status badge in the modal header immediately
                const statusBadge = $('#job-details-modal .status-badge');
                statusBadge.removeClass().addClass(`status-badge ${getStatusClass('ready')} text-base px-4 py-2`);
                statusBadge.html(`<i class="${getStatusIcon('ready')} mr-2"></i>${getStatusText('ready')}`);
            }
            
            // Show success notification with invoice details
            showNotification(`Job #${jobId} marked as ready! Invoice ${invoiceData.invoiceNumber} created in Xero.`, 'success');
            
            // Add invoice to the invoices list
            if (allInvoices) {
                allInvoices.push({
                    id: invoiceData.invoiceId,
                    jobId: jobId,
                    customer: invoiceData.customer,
                    description: invoiceData.description,
                    regNo: job.regNo,
                    amount: invoiceData.amount,
                    status: 'pending',
                    issueDate: invoiceData.createdDate,
                    dueDate: invoiceData.dueDate,
                    paidDate: null,
                    paymentMethod: null
                });
            }
            
            // Refresh the job queue and jobs table
            setTimeout(() => {
                displayJobQueue();
                displayJobsTable();
                // Refresh job details modal if it's open for this job
                if (window.currentJobDetails && window.currentJobDetails.id == jobId) {
                    showJobDetails(jobId);
                }
            }, 1000);
        })
        .catch(error => {
            console.error('Failed to create invoice:', error);
            showNotification(`Job #${jobId} marked as ready, but invoice creation failed: ${error.message}`, 'warning');
            
            // Still update the job status even if invoice creation fails
            job.status = 'ready';
            
            // Update job status in the current job details if it matches
            if (window.currentJobDetails && window.currentJobDetails.id == jobId) {
                window.currentJobDetails.status = 'ready';
                
                // Update the status badge in the modal header immediately
                const statusBadge = $('#job-details-modal .status-badge');
                statusBadge.removeClass().addClass(`status-badge ${getStatusClass('ready')} text-base px-4 py-2`);
                statusBadge.html(`<i class="${getStatusIcon('ready')} mr-2"></i>${getStatusText('ready')}`);
            }
            
            // Refresh the job queue and jobs table
            setTimeout(() => {
                displayJobQueue();
                displayJobsTable();
                // Refresh job details modal if it's open for this job
                if (window.currentJobDetails && window.currentJobDetails.id == jobId) {
                    showJobDetails(jobId);
                }
            }, 1000);
        });
}

function markJobAsInvoiced(jobId) {
    console.log('Marking job as invoiced:', jobId);
    showNotification(`Job #${jobId} marked as invoiced!`, 'success');
    
    // Update job status in the current job details if it matches
    if (window.currentJobDetails && window.currentJobDetails.id == jobId) {
        window.currentJobDetails.status = 'invoiced';
        
        // Update the status badge in the modal header immediately
        const statusBadge = $('#job-details-modal .status-badge');
        statusBadge.removeClass().addClass(`status-badge ${getStatusClass('invoiced')} text-base px-4 py-2`);
        statusBadge.html(`<i class="${getStatusIcon('invoiced')} mr-2"></i>${getStatusText('invoiced')}`);
    }
    
    // Refresh the job queue and jobs table
    setTimeout(() => {
        displayJobQueue();
        displayJobsTable();
        // Refresh job details modal if it's open for this job
        if (window.currentJobDetails && window.currentJobDetails.id == jobId) {
            showJobDetails(jobId);
        }
    }, 1000);
}

function addNewNote(noteText, jobId) {
    console.log('Adding new note:', noteText, 'for job:', jobId);
    
    // Create new note object
    const newNote = {
        id: Date.now(), // Simple ID generation
        text: noteText,
        author: 'Current User', // In a real app, this would be the logged-in user
        timestamp: new Date().toISOString(),
        type: 'general' // Default note type
    };
    
    // Add note to current job details
    if (window.currentJobDetails) {
        if (!window.currentJobDetails.notes) {
            window.currentJobDetails.notes = [];
        }
        window.currentJobDetails.notes.unshift(newNote); // Add to beginning of array
        
        // Show success notification
        showNotification('Note added successfully!', 'success');
        
        // Refresh the notes tab content
        refreshNotesTab();
        
        // Update the notes badge in the job queue and table
        setTimeout(() => {
            displayJobQueue();
            displayJobsTable();
        }, 500);
    }
}

function refreshNotesTab() {
    if (!window.currentJobDetails) return;
    
    const jobDetails = window.currentJobDetails;
    const notesHtml = jobDetails.notes && jobDetails.notes.length > 0 ? 
        jobDetails.notes.map(note => `
            <div class="note-item note-${note.type}">
                <div class="note-text">${note.text}</div>
                <div class="note-meta">
                    <span class="note-author">${note.author}</span>
                    <span class="note-timestamp">${formatDateTime(note.timestamp)}</span>
                </div>
            </div>
        `).join('') : 
        '<div class="text-center text-gray-500 py-8"><i class="fas fa-sticky-note text-4xl mb-2"></i><p>No notes yet</p></div>';
    
    $('.notes-container').html(notesHtml);
}

function enlargePhoto(photoSrc) {
    // Create photo modal
    const photoModal = $(`
        <div id="photo-modal" class="fixed inset-0 bg-black bg-opacity-90 z-50 flex items-center justify-center p-4">
            <div class="relative max-w-4xl max-h-full">
                <img src="${photoSrc}" alt="Enlarged photo" class="max-w-full max-h-full object-contain rounded-lg">
                <button id="close-photo-modal" class="absolute top-4 right-4 w-10 h-10 bg-black bg-opacity-50 hover:bg-opacity-70 text-white rounded-full flex items-center justify-center transition-all">
                    <i class="fas fa-times"></i>
                </button>
                <button id="download-photo-modal" class="absolute top-4 right-16 w-10 h-10 bg-black bg-opacity-50 hover:bg-opacity-70 text-white rounded-full flex items-center justify-center transition-all" data-photo="${photoSrc}" data-filename="${photoSrc.split('/').pop()}">
                    <i class="fas fa-download"></i>
                </button>
            </div>
        </div>
    `);
    
    $('body').append(photoModal);
    
    // Close modal functionality
    $('#close-photo-modal').click(function() {
        $('#photo-modal').remove();
    });
    
    // Download from modal
    $('#download-photo-modal').click(function() {
        const photoSrc = $(this).data('photo');
        const filename = $(this).data('filename');
        downloadPhoto(photoSrc, filename);
    });
    
    // Close on background click
    $('#photo-modal').click(function(e) {
        if (e.target === this) {
            $('#photo-modal').remove();
        }
    });
    
    // Close on escape key
    $(document).keydown(function(e) {
        if (e.key === 'Escape') {
            $('#photo-modal').remove();
        }
    });
}

function downloadPhoto(photoSrc, filename) {
    // Create a temporary link element
    const link = document.createElement('a');
    link.href = photoSrc;
    link.download = filename;
    
    // Append to body, click, and remove
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    // Show success notification
    showNotification(`Photo "${filename}" downloaded successfully!`, 'success');
}

function addNewAddon(addonData, jobId) {
    console.log('Adding new add-on:', addonData, 'for job:', jobId);
    
    // Create new add-on object
    const newAddon = {
        id: Date.now(), // Simple ID generation
        type: addonData.type,
        description: addonData.description,
        quantity: addonData.quantity,
        price: addonData.price,
        addedAt: new Date().toISOString()
    };
    
    // Add add-on to current job details
    if (window.currentJobDetails) {
        if (!window.currentJobDetails.addons) {
            window.currentJobDetails.addons = [];
        }
        window.currentJobDetails.addons.push(newAddon);
        
        // Show success notification
        showNotification(`Add-on "${addonData.type}" added successfully!`, 'success');
        
        // Refresh the add-ons container
        refreshAddonsContainer();
    }
}

function removeAddon(addonId, jobId) {
    console.log('Removing add-on:', addonId, 'from job:', jobId);
    
    if (window.currentJobDetails && window.currentJobDetails.addons) {
        const addonIndex = window.currentJobDetails.addons.findIndex(addon => addon.id == addonId);
        if (addonIndex > -1) {
            const removedAddon = window.currentJobDetails.addons.splice(addonIndex, 1)[0];
            showNotification(`Add-on "${removedAddon.type}" removed successfully!`, 'success');
            refreshAddonsContainer();
        }
    }
}

function refreshAddonsContainer() {
    if (!window.currentJobDetails) return;
    
    const jobDetails = window.currentJobDetails;
    const addonsHtml = jobDetails.addons && jobDetails.addons.length > 0 ? 
        jobDetails.addons.map(addon => `
            <div class="addon-item">
                <div class="flex justify-between items-start">
                    <div class="flex-1">
                        <h4 class="font-medium text-gray-900">${addon.type}</h4>
                        <p class="text-sm text-gray-600">${addon.description}</p>
                        <div class="flex items-center space-x-4 mt-1">
                            <span class="text-xs text-gray-500">Qty: ${addon.quantity}</span>
                            <span class="text-sm font-medium text-gray-900">£${addon.price.toFixed(2)}</span>
                        </div>
                    </div>
                    <button class="remove-addon-btn text-red-500 hover:text-red-700 transition-colors" data-addon-id="${addon.id}">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
        `).join('') : 
        '<div class="text-center text-gray-500 py-8"><i class="fas fa-plus-circle text-4xl mb-2"></i><p>No add-ons yet</p></div>';
    
    $('.addons-container').html(addonsHtml);
    
    // Update the add-ons tab badge
    const addonsCount = jobDetails.addons ? jobDetails.addons.length : 0;
    const addonsTab = $('.job-tab[data-tab="addons"]');
    if (addonsCount > 0) {
        if (addonsTab.find('.note-badge').length === 0) {
            addonsTab.append(`<span class="note-badge ml-2">${addonsCount}</span>`);
        } else {
            addonsTab.find('.note-badge').text(addonsCount);
        }
    } else {
        addonsTab.find('.note-badge').remove();
    }
    
    // Re-attach event listeners for remove buttons
    $('.remove-addon-btn').off('click').on('click', function() {
        const addonId = $(this).data('addon-id');
        removeAddon(addonId, window.currentJobDetails.id);
    });
}

function clearAddonForm() {
    $('#addon-type').val('');
    $('#addon-description').val('');
    $('#addon-quantity').val('1');
    $('#addon-price').val('');
}

// Create Job Functions
function createNewJob() {
    // Get form data
    const formData = {
        customer: $('#customer-name').val(),
        phone: $('#customer-phone').val(),
        email: $('#customer-email').val(),
        regNo: $('#reg-no').val(),
        description: $('#job-description').val(),
        quantity: parseInt($('#wheel-quantity').val()),
        serviceType: $('#service-type').val(),
        estimatedQuote: parseFloat($('#estimated-quote').val()) || 0,
        notes: $('#job-notes').val(),
        photos: []
    };
    
    // Validate required fields
    if (!formData.customer || !formData.description || !formData.quantity || !formData.serviceType) {
        showNotification('Please fill in all required fields', 'error');
        return;
    }
    
    // Generate new job ID (in real app, this would come from server)
    const newJobId = Date.now();
    
    // Create job object
    const newJob = {
        id: newJobId,
        customer: formData.customer,
        description: formData.description,
        regNo: formData.regNo,
        status: 'pending',
        priority: 'medium', // Default priority
        date: new Date().toISOString().split('T')[0],
        dueDate: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000).toISOString().split('T')[0], // 7 days from now
        quantity: formData.quantity,
        urgency: 'standard',
        quote: formData.estimatedQuote,
        notes: formData.notes ? [{
            id: 1,
            text: formData.notes,
            author: 'Staff Member',
            timestamp: new Date().toISOString(),
            type: 'general'
        }] : [],
        photos: formData.photos,
        assignedTo: 'Staff Member'
    };
    
    // In a real app, this would be sent to the server
    console.log('Creating new job:', newJob);
    
    // Show success message
    showNotification('Job created successfully!', 'success');
    
    // Close modal and clear form
    $('#create-job-modal').addClass('hidden');
    clearCreateJobForm();
    
    // Refresh the job queue (in real app, this would reload from server)
    // For demo purposes, we'll just show a message
    setTimeout(() => {
        showNotification('Job has been added to the queue', 'info');
    }, 1000);
}

function clearCreateJobForm() {
    $('#create-job-form')[0].reset();
    $('#photo-preview').addClass('hidden').empty();
    $('#reg-no').removeClass('reg-plate');
    $('#customer-details').addClass('hidden');
    $('#customer-select').val('');
}

function handlePhotoPreview(input) {
    const preview = $('#photo-preview');
    preview.empty();
    
    if (input.files && input.files.length > 0) {
        preview.removeClass('hidden');
        
        Array.from(input.files).forEach((file, index) => {
            if (file.type.startsWith('image/')) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    const photoItem = $(`
                        <div class="relative">
                            <img src="${e.target.result}" alt="Preview ${index + 1}" class="w-full h-20 object-cover rounded-lg">
                            <button type="button" class="absolute -top-2 -right-2 w-6 h-6 bg-red-500 text-white rounded-full flex items-center justify-center text-xs hover:bg-red-600" onclick="removePhotoPreview(${index})">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    `);
                    preview.append(photoItem);
                };
                reader.readAsDataURL(file);
            }
        });
    } else {
        preview.addClass('hidden');
    }
}

function removePhotoPreview(index) {
    // This would need to be implemented to remove specific photos
    // For now, we'll just clear all previews
    $('#photo-preview').addClass('hidden').empty();
    $('#job-photos').val('');
}

// Customer data
const customerData = {
    'direct-line': {
        name: 'Direct Line Insurance',
        phone: '+44 20 7123 4567',
        email: 'claims@directline.com'
    },
    'aviva': {
        name: 'Aviva Insurance',
        phone: '+44 20 7123 4568',
        email: 'claims@aviva.com'
    },
    'admiral': {
        name: 'Admiral Insurance',
        phone: '+44 20 7123 4569',
        email: 'claims@admiral.com'
    },
    'churchill': {
        name: 'Churchill Insurance',
        phone: '+44 20 7123 4570',
        email: 'claims@churchill.com'
    },
    'lv': {
        name: 'LV= Insurance',
        phone: '+44 20 7123 4571',
        email: 'claims@lv.com'
    }
};

function handleCustomerSelection() {
    const selectedCustomer = $('#customer-select').val();
    const customerDetails = $('#customer-details');
    
    if (selectedCustomer === 'new') {
        // Show add new customer form
        showAddCustomerForm();
    } else if (selectedCustomer && customerData[selectedCustomer]) {
        // Auto-fill customer details
        const customer = customerData[selectedCustomer];
        $('#customer-name').val(customer.name);
        $('#customer-phone').val(customer.phone);
        $('#customer-email').val(customer.email);
        customerDetails.removeClass('hidden');
    } else if (selectedCustomer) {
        // Show customer details form for manual entry
        customerDetails.removeClass('hidden');
        // Clear fields
        $('#customer-name').val('');
        $('#customer-phone').val('');
        $('#customer-email').val('');
    } else {
        // Hide customer details
        customerDetails.addClass('hidden');
    }
}

function showAddCustomerForm() {
    // Select "Add New Customer" option
    $('#customer-select').val('new');
    // Show customer details form
    $('#customer-details').removeClass('hidden');
    // Clear fields for new customer
    $('#customer-name').val('');
    $('#customer-phone').val('');
    $('#customer-email').val('');
    $('#reg-no').val('');
    // Focus on customer name
    $('#customer-name').focus();
}

// Customer Table Functions
function loadCustomersTable() {
    // This would load customers from the server
    // For now, we'll use demo data
    let customers = [
        {
            id: 1,
            name: 'Direct Line Insurance',
            phone: '+44 20 7123 4567',
            email: 'claims@directline.com',
            jobCount: 5,
            lastJob: '2024-01-15',
            status: 'active'
        },
        {
            id: 2,
            name: 'Aviva Insurance',
            phone: '+44 20 7123 4568',
            email: 'claims@aviva.com',
            jobCount: 3,
            lastJob: '2024-01-14',
            status: 'active'
        },
        {
            id: 3,
            name: 'Admiral Insurance',
            phone: '+44 20 7123 4569',
            email: 'claims@admiral.com',
            jobCount: 7,
            lastJob: '2024-01-12',
            status: 'active'
        },
        {
            id: 4,
            name: 'Churchill Insurance',
            phone: '+44 20 7123 4570',
            email: 'claims@churchill.com',
            jobCount: 2,
            lastJob: '2024-01-13',
            status: 'active'
        },
        {
            id: 5,
            name: 'LV= Insurance',
            phone: '+44 20 7123 4571',
            email: 'claims@lv.com',
            jobCount: 4,
            lastJob: '2024-01-11',
            status: 'active'
        },
        {
            id: 6,
            name: 'AXA Insurance',
            phone: '+44 20 7123 4572',
            email: 'claims@axa.com',
            jobCount: 6,
            lastJob: '2024-01-10',
            status: 'active'
        },
        {
            id: 7,
            name: 'Allianz Insurance',
            phone: '+44 20 7123 4573',
            email: 'claims@allianz.com',
            jobCount: 8,
            lastJob: '2024-01-09',
            status: 'active'
        },
        {
            id: 8,
            name: 'Zurich Insurance',
            phone: '+44 20 7123 4574',
            email: 'claims@zurich.com',
            jobCount: 3,
            lastJob: '2024-01-08',
            status: 'active'
        },
        {
            id: 9,
            name: 'Hastings Direct',
            phone: '+44 20 7123 4575',
            email: 'claims@hastingsdirect.com',
            jobCount: 4,
            lastJob: '2024-01-07',
            status: 'active'
        },
        {
            id: 10,
            name: 'More Than Insurance',
            phone: '+44 20 7123 4576',
            email: 'claims@morethan.com',
            jobCount: 5,
            lastJob: '2024-01-06',
            status: 'active'
        },
        {
            id: 11,
            name: 'Tesco Bank Insurance',
            phone: '+44 20 7123 4577',
            email: 'claims@tescobank.com',
            jobCount: 2,
            lastJob: '2024-01-05',
            status: 'active'
        },
        {
            id: 12,
            name: 'Sainsbury\'s Bank Insurance',
            phone: '+44 20 7123 4578',
            email: 'claims@sainsburysbank.co.uk',
            jobCount: 3,
            lastJob: '2024-01-04',
            status: 'active'
        },
        {
            id: 13,
            name: 'Co-op Insurance',
            phone: '+44 20 7123 4579',
            email: 'claims@coop.co.uk',
            jobCount: 4,
            lastJob: '2024-01-03',
            status: 'active'
        },
        {
            id: 14,
            name: 'NFU Mutual',
            phone: '+44 20 7123 4580',
            email: 'claims@nfumutual.co.uk',
            jobCount: 6,
            lastJob: '2024-01-02',
            status: 'active'
        },
        {
            id: 15,
            name: 'Privilege Insurance',
            phone: '+44 20 7123 4581',
            email: 'claims@privilege.com',
            jobCount: 3,
            lastJob: '2024-01-01',
            status: 'active'
        },
        {
            id: 16,
            name: 'Esure Insurance',
            phone: '+44 20 7123 4582',
            email: 'claims@esure.com',
            jobCount: 5,
            lastJob: '2023-12-31',
            status: 'active'
        },
        {
            id: 17,
            name: 'Sheilas\' Wheels',
            phone: '+44 20 7123 4583',
            email: 'claims@sheilaswheels.com',
            jobCount: 2,
            lastJob: '2023-12-30',
            status: 'active'
        },
        {
            id: 18,
            name: '1st Central Insurance',
            phone: '+44 20 7123 4584',
            email: 'claims@1stcentral.com',
            jobCount: 4,
            lastJob: '2023-12-29',
            status: 'active'
        },
        {
            id: 19,
            name: 'GoCompare Insurance',
            phone: '+44 20 7123 4585',
            email: 'claims@gocompare.com',
            jobCount: 3,
            lastJob: '2023-12-28',
            status: 'active'
        },
        {
            id: 20,
            name: 'Confused.com Insurance',
            phone: '+44 20 7123 4586',
            email: 'claims@confused.com',
            jobCount: 2,
            lastJob: '2023-12-27',
            status: 'active'
        }
    ];
    
    // Add sample pricing data to some customers
    customers = addSamplePricingData(customers);
    
    displayCustomersTable(customers);
}

function displayCustomersTable(customers) {
    const tbody = $('#customers-table-body');
    tbody.empty();
    
    if (customers.length === 0) {
        tbody.html(`
            <tr>
                <td colspan="6" class="px-6 py-12 text-center text-gray-500">
                    <div class="w-16 h-16 bg-gray-100 rounded-2xl flex items-center justify-center mx-auto mb-4">
                        <i class="fas fa-users text-gray-400 text-2xl"></i>
                    </div>
                    <h3 class="text-lg font-semibold text-gray-900 mb-2">No customers found</h3>
                    <p class="text-gray-600">No customers match your search criteria</p>
                </td>
            </tr>
        `);
        return;
    }
    
    customers.forEach(customer => {
        const row = createCustomerTableRow(customer);
        tbody.append(row);
    });
}

function createCustomerTableRow(customer) {
    const lastJobDate = new Date(customer.lastJob).toLocaleDateString('en-GB', {
        day: '2-digit',
        month: 'short',
        year: '2-digit'
    });
    
    return $(`
        <tr class="hover:bg-gray-50 transition-colors">
            <td class="px-6 py-4 whitespace-nowrap">
                <div class="flex items-center">
                    <div class="w-10 h-10 bg-gradient-to-br from-[#00a9ea]/10 to-[#00a9ea]/20 rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-building text-[#00a9ea]"></i>
                    </div>
                    <div>
                        <div class="text-sm font-medium text-gray-900">${customer.name}</div>
                        <div class="text-sm text-gray-500">ID: ${customer.id}</div>
                    </div>
                </div>
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                <div class="text-sm text-gray-900">${customer.phone}</div>
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                <div class="text-sm text-gray-900">${customer.email}</div>
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                    ${customer.jobCount} jobs
                </span>
            </td>
            <td class="px-6 py-4 whitespace-nowrap">
                <div class="text-sm text-gray-900">
                    <div class="flex items-center space-x-2">
                        <span class="text-xs text-gray-500">PC:</span>
                        <span class="font-medium">£${getCustomerServicePrice(customer, 'powderCoating')}</span>
                    </div>
                    <div class="flex items-center space-x-2">
                        <span class="text-xs text-gray-500">DC:</span>
                        <span class="font-medium">£${getCustomerServicePrice(customer, 'diamondCutting')}</span>
                    </div>
                    <div class="flex items-center space-x-2">
                        <span class="text-xs text-gray-500">Other:</span>
                        <span class="font-medium">£${getCustomerServicePrice(customer, 'otherServices')}</span>
                    </div>
                </div>
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                ${lastJobDate}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                <div class="flex items-center space-x-2">
                    <button class="text-[#00a9ea] hover:text-[#0088cc] transition-colors" title="View Details" onclick="viewCustomerDetails(${customer.id})">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="text-green-600 hover:text-green-900 transition-colors" title="Edit Customer" onclick="editCustomer(${customer.id})">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="text-purple-600 hover:text-purple-900 transition-colors" title="Edit Prices" onclick="editCustomerPrices(${customer.id})">
                        <i class="fas fa-pound-sign"></i>
                    </button>
                    <button class="text-red-600 hover:text-red-900 transition-colors" title="Delete Customer" onclick="deleteCustomer(${customer.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </td>
        </tr>
    `);
}

function filterCustomers(searchTerm) {
    // This would filter customers based on search term
    // For now, we'll just reload the table
    loadCustomersTable();
}

// Customer Action Functions
let currentEditingCustomer = null;

function editCustomer(customerId) {
    // Find the customer data
    const customers = getCustomersData();
    const customer = customers.find(c => c.id === customerId);
    
    if (!customer) {
        showNotification('Customer not found', 'error');
        return;
    }
    
    // Store the current editing customer
    currentEditingCustomer = customer;
    
    // Populate the edit form
    $('#edit-customer-name').val(customer.name);
    $('#edit-customer-phone').val(customer.phone);
    $('#edit-customer-email').val(customer.email);
    $('#edit-customer-status').val(customer.status);
    $('#edit-customer-notes').val(customer.notes || '');
    $('#edit-contact-person').val(customer.contactPerson || '');
    $('#edit-account-manager').val(customer.accountManager || '');
    
    
    // Show the modal
    $('#edit-customer-modal').removeClass('hidden');
}

function saveCustomerChanges() {
    if (!currentEditingCustomer) {
        showNotification('No customer selected for editing', 'error');
        return;
    }
    
    // Get form data
    const formData = {
        name: $('#edit-customer-name').val(),
        phone: $('#edit-customer-phone').val(),
        email: $('#edit-customer-email').val(),
        status: $('#edit-customer-status').val(),
        notes: $('#edit-customer-notes').val(),
        contactPerson: $('#edit-contact-person').val(),
        accountManager: $('#edit-account-manager').val()
    };
    
    // Validate required fields
    if (!formData.name) {
        showNotification('Customer name is required', 'error');
        return;
    }
    
    // Update the customer data (in a real app, this would be sent to the server)
    currentEditingCustomer.name = formData.name;
    currentEditingCustomer.phone = formData.phone;
    currentEditingCustomer.email = formData.email;
    currentEditingCustomer.status = formData.status;
    currentEditingCustomer.notes = formData.notes;
    currentEditingCustomer.contactPerson = formData.contactPerson;
    currentEditingCustomer.accountManager = formData.accountManager;
    
    // Show success message
    showNotification('Customer updated successfully!', 'success');
    
    // Close modal
    $('#edit-customer-modal').addClass('hidden');
    
    // Refresh the table
    loadCustomersTable();
    
    // Clear current editing customer
    currentEditingCustomer = null;
}

function viewCustomerDetails(customerId) {
    // This would open a customer details modal
    showNotification('View customer details feature coming soon!', 'info');
}

function deleteCustomer(customerId) {
    // This would show a confirmation dialog and delete the customer
    if (confirm('Are you sure you want to delete this customer? This action cannot be undone.')) {
        showNotification('Customer deleted successfully!', 'success');
        // In a real app, this would delete from the server
        loadCustomersTable();
    }
}

function getDefaultServicePrices() {
    // Get default prices from settings or use hardcoded defaults
    const savedSettings = localStorage.getItem('appSettings');
    if (savedSettings) {
        try {
            const settings = JSON.parse(savedSettings);
            if (settings.services) {
                return {
                    powderCoating: settings.services.powderCoatingPrice || 120,
                    diamondCutting: settings.services.diamondCuttingPrice || 150,
                    otherServices: settings.services.otherServicesPrice || 100
                };
            }
        } catch (e) {
            console.error('Error loading default prices:', e);
        }
    }
    
    // Fallback to hardcoded defaults
    return {
        powderCoating: 120,
        diamondCutting: 150,
        otherServices: 100
    };
}

function getCustomerServicePrice(customer, serviceType) {
    // Get customer-specific price or default price
    const defaultPrices = getDefaultServicePrices();
    
    switch (serviceType) {
        case 'powderCoating':
            return customer.powderCoatingPrice || defaultPrices.powderCoating;
        case 'diamondCutting':
            return customer.diamondCuttingPrice || defaultPrices.diamondCutting;
        case 'otherServices':
            return customer.otherServicesPrice || defaultPrices.otherServices;
        default:
            return 0;
    }
}

function addSamplePricingData(customers) {
    // Add sample pricing data to some customers to demonstrate the feature
    const customersWithPricing = customers.map(customer => {
        switch (customer.id) {
            case 1: // Direct Line Insurance
                return {
                    ...customer,
                    powderCoatingPrice: 110, // 10% discount
                    diamondCuttingPrice: 140, // 10% discount
                    otherServicesPrice: 95 // 5% discount
                };
            case 2: // Aviva Insurance
                return {
                    ...customer,
                    powderCoatingPrice: 115, // 5% discount
                    diamondCuttingPrice: 145, // 5% discount
                    // otherServicesPrice uses default
                };
            case 3: // Admiral Insurance
                return {
                    ...customer,
                    powderCoatingPrice: 105, // 15% discount for high volume
                    diamondCuttingPrice: 135, // 15% discount for high volume
                    otherServicesPrice: 90 // 10% discount for high volume
                };
            case 7: // Allianz Insurance
                return {
                    ...customer,
                    powderCoatingPrice: 108, // 12% discount
                    diamondCuttingPrice: 138, // 12% discount
                    otherServicesPrice: 92 // 8% discount
                };
            default:
                return customer; // Use default pricing
        }
    });
    
    return customersWithPricing;
}

// Customer Pricing Modal Functions
let currentPricingCustomer = null;

function editCustomerPrices(customerId) {
    // Find the customer data
    const customers = getCustomersData();
    const customer = customers.find(c => c.id === customerId);
    
    if (!customer) {
        showNotification('Customer not found', 'error');
        return;
    }
    
    // Store the current pricing customer
    currentPricingCustomer = customer;
    
    // Update modal title
    $('#customer-prices-title').text(`Set custom pricing for ${customer.name}`);
    
    // Get default prices
    const defaultPrices = getDefaultServicePrices();
    
    // Update default price displays
    $('#default-powder-coating').text(defaultPrices.powderCoating);
    $('#default-diamond-cutting').text(defaultPrices.diamondCutting);
    $('#default-other-services').text(defaultPrices.otherServices);
    
    // Set up form with current customer pricing
    setupPricingForm(customer, defaultPrices);
    
    // Show the modal
    $('#edit-customer-prices-modal').removeClass('hidden');
}

function setupPricingForm(customer, defaultPrices) {
    // Powder Coating
    if (customer.powderCoatingPrice) {
        $('#use-default-powder-coating').prop('checked', false);
        $('#edit-powder-coating-price').val(customer.powderCoatingPrice).prop('disabled', false);
    } else {
        $('#use-default-powder-coating').prop('checked', true);
        $('#edit-powder-coating-price').val('').prop('disabled', true);
    }
    
    // Diamond Cutting
    if (customer.diamondCuttingPrice) {
        $('#use-default-diamond-cutting').prop('checked', false);
        $('#edit-diamond-cutting-price').val(customer.diamondCuttingPrice).prop('disabled', false);
    } else {
        $('#use-default-diamond-cutting').prop('checked', true);
        $('#edit-diamond-cutting-price').val('').prop('disabled', true);
    }
    
    // Other Services
    if (customer.otherServicesPrice) {
        $('#use-default-other-services').prop('checked', false);
        $('#edit-other-services-price').val(customer.otherServicesPrice).prop('disabled', false);
    } else {
        $('#use-default-other-services').prop('checked', true);
        $('#edit-other-services-price').val('').prop('disabled', true);
    }
    
    // Update pricing summary
    updatePricingSummary();
}

function togglePriceInput(serviceType, useDefault) {
    const inputId = `#edit-${serviceType}-price`;
    const input = $(inputId);
    
    if (useDefault) {
        input.prop('disabled', true).val('');
    } else {
        input.prop('disabled', false).focus();
    }
    
    updatePricingSummary();
}

function updatePricingSummary() {
    const defaultPrices = getDefaultServicePrices();
    
    // Get current prices (custom or default)
    const powderCoatingPrice = $('#edit-powder-coating-price').val() || defaultPrices.powderCoating;
    const diamondCuttingPrice = $('#edit-diamond-cutting-price').val() || defaultPrices.diamondCutting;
    const otherServicesPrice = $('#edit-other-services-price').val() || defaultPrices.otherServices;
    
    // Update summary display
    $('#summary-powder-coating').text(`£${powderCoatingPrice}`);
    $('#summary-diamond-cutting').text(`£${diamondCuttingPrice}`);
    $('#summary-other-services').text(`£${otherServicesPrice}`);
}

function saveCustomerPrices() {
    if (!currentPricingCustomer) {
        showNotification('No customer selected for pricing', 'error');
        return;
    }
    
    // Get form data
    const defaultPrices = getDefaultServicePrices();
    
    const pricingData = {
        powderCoatingPrice: $('#use-default-powder-coating').is(':checked') ? null : parseFloat($('#edit-powder-coating-price').val()) || null,
        diamondCuttingPrice: $('#use-default-diamond-cutting').is(':checked') ? null : parseFloat($('#edit-diamond-cutting-price').val()) || null,
        otherServicesPrice: $('#use-default-other-services').is(':checked') ? null : parseFloat($('#edit-other-services-price').val()) || null
    };
    
    // Update the customer data
    currentPricingCustomer.powderCoatingPrice = pricingData.powderCoatingPrice;
    currentPricingCustomer.diamondCuttingPrice = pricingData.diamondCuttingPrice;
    currentPricingCustomer.otherServicesPrice = pricingData.otherServicesPrice;
    
    // Show success message
    showNotification('Customer pricing updated successfully!', 'success');
    
    // Close modal
    $('#edit-customer-prices-modal').addClass('hidden');
    
    // Refresh the table
    loadCustomersTable();
    
    // Clear current pricing customer
    currentPricingCustomer = null;
}

// Add Service Modal Functions
let servicesList = [
    {
        id: 1,
        name: 'Powder Coating',
        description: 'Wheel powder coating service',
        icon: 'fas fa-paint-brush',
        iconColor: 'blue',
        price: 120,
        priceType: 'per-wheel',
        active: true
    },
    {
        id: 2,
        name: 'Diamond Cutting',
        description: 'Precision diamond cutting service',
        icon: 'fas fa-cut',
        iconColor: 'purple',
        price: 150,
        priceType: 'per-wheel',
        active: true
    },
    {
        id: 3,
        name: 'Other Services',
        description: 'Custom wheel repair services',
        icon: 'fas fa-tools',
        iconColor: 'orange',
        price: 100,
        priceType: 'per-wheel',
        active: true
    }
];

function openAddServiceModal() {
    // Reset form
    $('#add-service-form')[0].reset();
    $('#service-active').prop('checked', true);
    
    // Update preview
    updateServicePreview();
    
    // Show modal
    $('#add-service-modal').removeClass('hidden');
}

function updateServicePreview() {
    const name = $('#service-name').val() || 'Service Name';
    const description = $('#service-description').val() || 'Service description';
    const icon = $('#service-icon').val() || 'fas fa-paint-brush';
    const iconColor = $('#service-icon-color').val() || 'blue';
    const price = $('#service-price').val() || '0.00';
    const priceType = $('#service-price-type').val() || 'per-wheel';
    
    // Update preview elements
    $('#preview-name').text(name);
    $('#preview-description').text(description);
    $('#preview-price').text(`£${parseFloat(price).toFixed(2)}`);
    $('#preview-type').text(priceType.replace('-', ' '));
    
    // Update icon
    const iconElement = $('#preview-icon i');
    iconElement.removeClass().addClass(icon);
    
    // Update icon color
    const colorClasses = {
        blue: 'bg-blue-100 text-blue-600',
        purple: 'bg-purple-100 text-purple-600',
        orange: 'bg-orange-100 text-orange-600',
        green: 'bg-green-100 text-green-600',
        red: 'bg-red-100 text-red-600',
        indigo: 'bg-indigo-100 text-indigo-600',
        pink: 'bg-pink-100 text-pink-600',
        yellow: 'bg-yellow-100 text-yellow-600'
    };
    
    $('#preview-icon').removeClass().addClass(`w-10 h-10 ${colorClasses[iconColor]} rounded-lg flex items-center justify-center mr-3`);
}

function saveNewService() {
    // Get form data
    const formData = {
        name: $('#service-name').val(),
        description: $('#service-description').val(),
        icon: $('#service-icon').val(),
        iconColor: $('#service-icon-color').val(),
        price: parseFloat($('#service-price').val()),
        priceType: $('#service-price-type').val(),
        active: $('#service-active').is(':checked')
    };
    
    // Validate required fields
    if (!formData.name) {
        showNotification('Service name is required', 'error');
        return;
    }
    
    if (!formData.price || formData.price <= 0) {
        showNotification('Valid price is required', 'error');
        return;
    }
    
    // Check if service name already exists
    const existingService = servicesList.find(service => 
        service.name.toLowerCase() === formData.name.toLowerCase()
    );
    
    if (existingService) {
        showNotification('A service with this name already exists', 'error');
        return;
    }
    
    // Create new service
    const newService = {
        id: servicesList.length + 1,
        ...formData
    };
    
    // Add to services list
    servicesList.push(newService);
    
    // Update settings with new service pricing
    updateSettingsWithNewService(newService);
    
    // Show success message
    showNotification('Service added successfully!', 'success');
    
    // Close modal
    $('#add-service-modal').addClass('hidden');
    
    // Refresh services display in settings
    refreshServicesDisplay();
}

function updateSettingsWithNewService(service) {
    // Get current settings
    const savedSettings = localStorage.getItem('appSettings');
    let settings = {};
    
    if (savedSettings) {
        try {
            settings = JSON.parse(savedSettings);
        } catch (e) {
            console.error('Error loading settings:', e);
        }
    }
    
    // Initialize services if not exists
    if (!settings.services) {
        settings.services = {};
    }
    
    // Add new service price to settings
    const serviceKey = service.name.toLowerCase().replace(/\s+/g, '') + 'Price';
    settings.services[serviceKey] = service.price;
    
    // Save updated settings
    localStorage.setItem('appSettings', JSON.stringify(settings));
}

function refreshServicesDisplay() {
    // This would refresh the services list in the settings page
    // For now, we'll just show a notification
    showNotification('Services list updated', 'info');
}

function getCustomersData() {
    // This would get customers from the server
    // For now, we'll return the demo data
    return [
        {
            id: 1,
            name: 'Direct Line Insurance',
            phone: '+44 20 7123 4567',
            email: 'claims@directline.com',
            jobCount: 5,
            lastJob: '2024-01-15',
            status: 'active',
            powderCoatingPrice: 110, // 10% discount
            diamondCuttingPrice: 140, // 10% discount
            otherServicesPrice: 95 // 5% discount
        },
        {
            id: 2,
            name: 'Aviva Insurance',
            phone: '+44 20 7123 4568',
            email: 'claims@aviva.com',
            jobCount: 3,
            lastJob: '2024-01-14',
            status: 'active'
        },
        {
            id: 3,
            name: 'Admiral Insurance',
            phone: '+44 20 7123 4569',
            email: 'claims@admiral.com',
            jobCount: 7,
            lastJob: '2024-01-12',
            status: 'active'
        },
        {
            id: 4,
            name: 'Churchill Insurance',
            phone: '+44 20 7123 4570',
            email: 'claims@churchill.com',
            jobCount: 2,
            lastJob: '2024-01-13',
            status: 'active'
        },
        {
            id: 5,
            name: 'LV= Insurance',
            phone: '+44 20 7123 4571',
            email: 'claims@lv.com',
            jobCount: 4,
            lastJob: '2024-01-11',
            status: 'active'
        },
        {
            id: 6,
            name: 'AXA Insurance',
            phone: '+44 20 7123 4572',
            email: 'claims@axa.com',
            jobCount: 6,
            lastJob: '2024-01-10',
            status: 'active'
        },
        {
            id: 7,
            name: 'Allianz Insurance',
            phone: '+44 20 7123 4573',
            email: 'claims@allianz.com',
            jobCount: 8,
            lastJob: '2024-01-09',
            status: 'active'
        },
        {
            id: 8,
            name: 'Zurich Insurance',
            phone: '+44 20 7123 4574',
            email: 'claims@zurich.com',
            jobCount: 3,
            lastJob: '2024-01-08',
            status: 'active'
        },
        {
            id: 9,
            name: 'Hastings Direct',
            phone: '+44 20 7123 4575',
            email: 'claims@hastingsdirect.com',
            jobCount: 4,
            lastJob: '2024-01-07',
            status: 'active'
        },
        {
            id: 10,
            name: 'More Than Insurance',
            phone: '+44 20 7123 4576',
            email: 'claims@morethan.com',
            jobCount: 5,
            lastJob: '2024-01-06',
            status: 'active'
        },
        {
            id: 11,
            name: 'Tesco Bank Insurance',
            phone: '+44 20 7123 4577',
            email: 'claims@tescobank.com',
            jobCount: 2,
            lastJob: '2024-01-05',
            status: 'active'
        },
        {
            id: 12,
            name: 'Sainsbury\'s Bank Insurance',
            phone: '+44 20 7123 4578',
            email: 'claims@sainsburysbank.co.uk',
            jobCount: 3,
            lastJob: '2024-01-04',
            status: 'active'
        },
        {
            id: 13,
            name: 'Co-op Insurance',
            phone: '+44 20 7123 4579',
            email: 'claims@coop.co.uk',
            jobCount: 4,
            lastJob: '2024-01-03',
            status: 'active'
        },
        {
            id: 14,
            name: 'NFU Mutual',
            phone: '+44 20 7123 4580',
            email: 'claims@nfumutual.co.uk',
            jobCount: 6,
            lastJob: '2024-01-02',
            status: 'active'
        },
        {
            id: 15,
            name: 'Privilege Insurance',
            phone: '+44 20 7123 4581',
            email: 'claims@privilege.com',
            jobCount: 3,
            lastJob: '2024-01-01',
            status: 'active'
        },
        {
            id: 16,
            name: 'Esure Insurance',
            phone: '+44 20 7123 4582',
            email: 'claims@esure.com',
            jobCount: 5,
            lastJob: '2023-12-31',
            status: 'active'
        },
        {
            id: 17,
            name: 'Sheilas\' Wheels',
            phone: '+44 20 7123 4583',
            email: 'claims@sheilaswheels.com',
            jobCount: 2,
            lastJob: '2023-12-30',
            status: 'active'
        },
        {
            id: 18,
            name: '1st Central Insurance',
            phone: '+44 20 7123 4584',
            email: 'claims@1stcentral.com',
            jobCount: 4,
            lastJob: '2023-12-29',
            status: 'active'
        },
        {
            id: 19,
            name: 'GoCompare Insurance',
            phone: '+44 20 7123 4585',
            email: 'claims@gocompare.com',
            jobCount: 3,
            lastJob: '2023-12-28',
            status: 'active'
        },
        {
            id: 20,
            name: 'Confused.com Insurance',
            phone: '+44 20 7123 4586',
            email: 'claims@confused.com',
            jobCount: 2,
            lastJob: '2023-12-27',
            status: 'active'
        }
    ];
}

// Settings Functions
function switchSettingsTab(tabName) {
    // Remove active class from all tabs
    $('.settings-tab').removeClass('active border-[#00a9ea] text-[#00a9ea]').addClass('border-transparent text-gray-500');
    
    // Hide all content
    $('.settings-content').addClass('hidden');
    
    if (tabName === 'general') {
        $('#general-settings-tab').removeClass('border-transparent text-gray-500').addClass('active border-[#00a9ea] text-[#00a9ea]');
        $('#general-settings-content').removeClass('hidden');
    } else if (tabName === 'services') {
        $('#services-settings-tab').removeClass('border-transparent text-gray-500').addClass('active border-[#00a9ea] text-[#00a9ea]');
        $('#services-settings-content').removeClass('hidden');
    } else if (tabName === 'xero') {
        $('#xero-settings-tab').removeClass('border-transparent text-gray-500').addClass('active border-[#00a9ea] text-[#00a9ea]');
        $('#xero-settings-content').removeClass('hidden');
        loadXeroSettings();
    }
}

function saveSettings() {
    // Collect all settings data
    const settings = {
        company: {
            name: $('#company-name').val(),
            email: $('#company-email').val(),
            phone: $('#company-phone').val(),
            address: $('#company-address').val()
        },
        application: {
            emailNotifications: $('#email-notifications').is(':checked'),
            autoSave: $('#auto-save').is(':checked'),
            darkMode: $('#dark-mode').is(':checked')
        },
        job: {
            defaultPriority: $('#default-priority').val(),
            autoAssign: $('#auto-assign').val()
        },
        services: {
            powderCoatingPrice: $('#powder-coating-price').val(),
            diamondCuttingPrice: $('#diamond-cutting-price').val(),
            otherServicesPrice: $('#other-services-price').val()
        }
    };
    
    // In a real app, this would be sent to the server
    console.log('Saving settings:', settings);
    
    // Show success notification
    showNotification('Settings saved successfully!', 'success');
    
    // Store in localStorage for demo purposes
    localStorage.setItem('appSettings', JSON.stringify(settings));
}

function resetSettings() {
    if (confirm('Are you sure you want to reset all settings to default values? This action cannot be undone.')) {
        // Reset form values to defaults
        $('#company-name').val('RefurbFlow Automotive Ltd');
        $('#company-email').val('info@RefurbFlow-automotive.com');
        $('#company-phone').val('+44 20 1234 5678');
        $('#company-address').val('123 Automotive Street, London, UK');
        
        $('#email-notifications').prop('checked', true);
        $('#auto-save').prop('checked', true);
        $('#dark-mode').prop('checked', false);
        
        $('#default-priority').val('medium');
        $('#auto-assign').val('auto');
        
        $('#powder-coating-price').val('120');
        $('#diamond-cutting-price').val('150');
        $('#other-services-price').val('100');
        
        showNotification('Settings reset to defaults', 'info');
    }
}

function loadSettings() {
    // Load settings from localStorage
    const savedSettings = localStorage.getItem('appSettings');
    if (savedSettings) {
        try {
            const settings = JSON.parse(savedSettings);
            
            // Apply company settings
            if (settings.company) {
                $('#company-name').val(settings.company.name || 'RefurbFlow Automotive Ltd');
                $('#company-email').val(settings.company.email || 'info@RefurbFlow-automotive.com');
                $('#company-phone').val(settings.company.phone || '+44 20 1234 5678');
                $('#company-address').val(settings.company.address || '123 Automotive Street, London, UK');
            }
            
            // Apply application settings
            if (settings.application) {
                $('#email-notifications').prop('checked', settings.application.emailNotifications !== false);
                $('#auto-save').prop('checked', settings.application.autoSave !== false);
                $('#dark-mode').prop('checked', settings.application.darkMode === true);
            }
            
            // Apply job settings
            if (settings.job) {
                $('#default-priority').val(settings.job.defaultPriority || 'medium');
                $('#auto-assign').val(settings.job.autoAssign || 'auto');
            }
            
            // Apply service pricing
            if (settings.services) {
                $('#powder-coating-price').val(settings.services.powderCoatingPrice || '120');
                $('#diamond-cutting-price').val(settings.services.diamondCuttingPrice || '150');
                $('#other-services-price').val(settings.services.otherServicesPrice || '100');
            }
        } catch (e) {
            console.error('Error loading settings:', e);
        }
    }
}

function markInvoiceAsPaid(invoiceNumber, jobId) {
    console.log('Mark as paid clicked for invoice:', invoiceNumber, 'job:', jobId);
    
    // Show immediate feedback
    showNotification(`Processing payment for invoice ${invoiceNumber}...`, 'info');
    
    // Find and update the invoice status
    const jobDetails = window.currentJobDetails || {};
    console.log('Current job details:', jobDetails);
    
    if (jobDetails.invoices) {
        const invoice = jobDetails.invoices.find(inv => inv.invoiceNumber === invoiceNumber);
        console.log('Found invoice:', invoice);
        
        if (invoice) {
            invoice.status = 'paid';
            console.log('Updated invoice status to paid');
            
            // Show success message with enhanced styling
            showNotification(`✅ Invoice ${invoiceNumber} marked as paid successfully!`, 'success');
            
            // Refresh the job details modal to show updated status
            setTimeout(() => {
                showJobDetails(jobId);
            }, 1000);
        } else {
            console.log('Invoice not found');
            showNotification('Invoice not found', 'error');
        }
    } else {
        console.log('No invoices found');
        showNotification('No invoices found', 'error');
    }
}

function editJob(jobId) {
    showNotification('Edit job feature coming soon!', 'info');
}

function updateJobStatus(jobId) {
    showNotification('Update status feature coming soon!', 'info');
}

function filterJobs() {
    const status = $('#status-filter').val();
    const customer = $('#customer-filter').val();
    const dateFrom = $('#date-from').val();
    
    // This would filter jobs based on criteria
    showNotification('Filtering jobs...', 'info');
    
    // Reload jobs table with filters
    setTimeout(() => {
        loadJobsTable();
        showNotification('Jobs filtered successfully', 'success');
    }, 1000);
}

function getStatusClass(status) {
    const statusMap = {
        'pending': 'status-pending',
        'in-progress': 'status-in-progress',
        'completed': 'status-completed',
        'invoiced': 'status-invoiced',
        'cancelled': 'status-cancelled',
        'ready': 'status-ready'
    };
    return statusMap[status] || 'status-pending';
}

function getStatusText(status) {
    const statusMap = {
        'pending': 'Pending',
        'in-progress': 'In Progress',
        'completed': 'Completed',
        'invoiced': 'Invoiced',
        'cancelled': 'Cancelled',
        'ready': 'Ready'
    };
    return statusMap[status] || 'Pending';
}

function getStatusIcon(status) {
    const iconMap = {
        'pending': 'fas fa-clock',
        'in-progress': 'fas fa-tools',
        'completed': 'fas fa-check-circle',
        'invoiced': 'fas fa-receipt',
        'cancelled': 'fas fa-times-circle',
        'ready': 'fas fa-check-double'
    };
    return iconMap[status] || 'fas fa-clock';
}

function getPriorityClass(priority) {
    const priorityMap = {
        'high': 'priority-high',
        'medium': 'priority-medium',
        'low': 'priority-low'
    };
    return priorityMap[priority] || 'priority-medium';
}

function getPriorityColor(priority) {
    const colorMap = {
        'high': '#dc2626', // Red
        'medium': '#f59e0b', // Amber
        'low': '#10b981' // Emerald
    };
    return colorMap[priority] || '#f59e0b'; // Default to amber
}

function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-GB', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

function showNotification(message, type = 'info') {
    const notification = $(`
        <div class="notification notification-${type} transform translate-x-full transition-transform duration-300">
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : type === 'error' ? 'fa-exclamation-circle' : type === 'warning' ? 'fa-exclamation-triangle' : 'fa-info-circle'} mr-3"></i>
                <span class="font-medium">${message}</span>
            </div>
        </div>
    `);
    
    $('body').append(notification);
    
    // Animate in
    setTimeout(() => {
        notification.removeClass('translate-x-full');
    }, 100);
    
    // Auto remove after 4 seconds for success messages, 3 seconds for others
    const duration = type === 'success' ? 4000 : 3000;
    setTimeout(() => {
        notification.addClass('translate-x-full');
        setTimeout(() => {
            notification.remove();
        }, 300);
    }, duration);
}

// Activities Page Functions
function loadActivities() {
    // Show loading state
    $('#activities-loading').removeClass('hidden');
    $('#activities-list').empty();
    $('#no-activities').addClass('hidden');
    
    // Load activities data
    $.ajax({
        url: 'data/activities.json',
        method: 'GET',
        dataType: 'json',
        success: function(data) {
            displayActivities(data.activities);
            setupActivityFilters();
        },
        error: function() {
            // Fallback to demo data
            const demoActivities = [
                {
                    id: 1,
                    type: "job",
                    action: "created",
                    description: "New job created for BMW 18\" Alloy Wheel Refurbishment",
                    user: "admin",
                    userName: "Admin User",
                    timestamp: "2024-01-15T10:30:00Z",
                    details: {
                        jobId: 1,
                        customer: "Direct Line Insurance",
                        service: "Powder Coating",
                        regNo: "BM18 ABC"
                    },
                    icon: "fas fa-plus-circle",
                    color: "green"
                },
                {
                    id: 2,
                    type: "job",
                    action: "status_changed",
                    description: "Job status changed from Pending to In Progress",
                    user: "staff",
                    userName: "John Smith",
                    timestamp: "2024-01-15T11:15:00Z",
                    details: {
                        jobId: 1,
                        oldStatus: "pending",
                        newStatus: "in-progress"
                    },
                    icon: "fas fa-arrow-right",
                    color: "blue"
                },
                {
                    id: 3,
                    type: "customer",
                    action: "updated",
                    description: "Customer information updated for Aviva Insurance",
                    user: "admin",
                    userName: "Admin User",
                    timestamp: "2024-01-15T14:20:00Z",
                    details: {
                        customerId: 2,
                        customerName: "Aviva Insurance",
                        changes: ["contact_email", "phone_number"]
                    },
                    icon: "fas fa-user-edit",
                    color: "orange"
                }
            ];
            displayActivities(demoActivities);
            setupActivityFilters();
        }
    });
}

function displayActivities(activities) {
    const activitiesList = $('#activities-list');
    const loadingElement = $('#activities-loading');
    const noActivitiesElement = $('#no-activities');
    
    // Hide loading
    loadingElement.addClass('hidden');
    
    if (!activities || activities.length === 0) {
        noActivitiesElement.removeClass('hidden');
        return;
    }
    
    // Sort activities by timestamp (newest first)
    const sortedActivities = activities.sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp));
    
    // Clear existing activities
    activitiesList.empty();
    
    // Display activities
    sortedActivities.forEach(activity => {
        const activityElement = createActivityItem(activity);
        activitiesList.append(activityElement);
    });
}

function createActivityItem(activity) {
    const timeAgo = getTimeAgo(activity.timestamp);
    const userBadge = getUserBadge(activity.user);
    const typeBadge = getTypeBadge(activity.type);
    const userIcon = getUserIcon(activity.user);
    const userName = activity.userName || getUserDisplayName(activity.user);
    const userRole = getUserRole(activity.user);
    
    return $(`
        <div class="activity-item bg-gray-50 rounded-xl p-4 border border-gray-200 hover:shadow-md transition-all duration-200">
            <div class="flex items-start space-x-4">
                <div class="flex-shrink-0">
                    <div class="w-10 h-10 rounded-lg flex items-center justify-center" style="background-color: ${getActivityColor(activity.color)}20; color: ${getActivityColor(activity.color)};">
                        <i class="${activity.icon}"></i>
                    </div>
                </div>
                <div class="flex-1 min-w-0">
                    <div class="activity-performed-by">
                        <i class="${userIcon} text-xs"></i>
                        <span class="font-medium">${userName}</span>
                        <span class="text-gray-400">•</span>
                        <span class="text-xs">${userRole}</span>
                    </div>
                    <div class="flex items-center justify-between mb-3">
                        <div class="flex items-center space-x-2">
                            <h4 class="text-sm font-semibold text-gray-900">${activity.description}</h4>
                            ${typeBadge}
                        </div>
                        <div class="flex items-center space-x-2">
                            <span class="text-xs text-gray-500">${timeAgo}</span>
                            <div class="w-2 h-2 rounded-full activity-status-indicator" style="background-color: ${getActivityColor(activity.color)};"></div>
                        </div>
                    </div>
                    <div class="flex items-center justify-between">
                        <div class="activity-user-info">
                            <div class="activity-user-avatar" style="background-color: ${getUserColor(activity.user)}20; color: ${getUserColor(activity.user)};">
                                <i class="${userIcon} text-xs"></i>
                            </div>
                            <div class="activity-user-details">
                                <span class="activity-user-name">${userName}</span>
                                <span class="activity-user-role">${userRole}</span>
                            </div>
                        </div>
                        <div class="flex items-center space-x-2">
                            ${userBadge}
                            <span class="text-xs text-gray-500 activity-time">${formatActivityTime(activity.timestamp)}</span>
                        </div>
                        ${activity.details ? `<button class="text-xs text-[#00a9ea] hover:text-[#0088cc] font-medium activity-details-btn" onclick="showActivityDetails(${activity.id})">View Details</button>` : ''}
                    </div>
                </div>
            </div>
        </div>
    `);
}

function setupActivityFilters() {
    // Activity type filter
    $('#activity-type-filter').on('change', function() {
        filterActivities();
    });
    
    // User filter
    $('#user-filter').on('change', function() {
        filterActivities();
    });
    
    // Date range filter
    $('#date-range-filter').on('change', function() {
        filterActivities();
    });
    
    // Clear filters
    $('#clear-filters').on('click', function() {
        $('#activity-type-filter, #user-filter, #date-range-filter').val('');
        filterActivities();
    });
}

function filterActivities() {
    const typeFilter = $('#activity-type-filter').val();
    const userFilter = $('#user-filter').val();
    const dateFilter = $('#date-range-filter').val();
    
    // This would implement filtering logic
    // For now, just reload all activities
    loadActivities();
}

function getTimeAgo(timestamp) {
    const now = new Date();
    const activityTime = new Date(timestamp);
    const diffInSeconds = Math.floor((now - activityTime) / 1000);
    
    if (diffInSeconds < 60) {
        return 'Just now';
    } else if (diffInSeconds < 3600) {
        const minutes = Math.floor(diffInSeconds / 60);
        return `${minutes}m ago`;
    } else if (diffInSeconds < 86400) {
        const hours = Math.floor(diffInSeconds / 3600);
        return `${hours}h ago`;
    } else {
        const days = Math.floor(diffInSeconds / 86400);
        return `${days}d ago`;
    }
}

function getUserBadge(user) {
    const userMap = {
        'admin': { text: 'Admin', color: 'bg-red-100 text-red-800' },
        'staff': { text: 'Staff', color: 'bg-blue-100 text-blue-800' },
        'system': { text: 'System', color: 'bg-gray-100 text-gray-800' }
    };
    
    const userInfo = userMap[user] || { text: user, color: 'bg-gray-100 text-gray-800' };
    
    return `<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${userInfo.color}">${userInfo.text}</span>`;
}

function getUserIcon(user) {
    const iconMap = {
        'admin': 'fas fa-user-shield',
        'staff': 'fas fa-user',
        'system': 'fas fa-cog'
    };
    
    return iconMap[user] || 'fas fa-user';
}

function getUserColor(user) {
    const colorMap = {
        'admin': '#dc2626', // Red
        'staff': '#3b82f6', // Blue
        'system': '#6b7280' // Gray
    };
    
    return colorMap[user] || '#6b7280';
}

function getUserDisplayName(user) {
    const nameMap = {
        'admin': 'Admin User',
        'staff': 'Staff Member',
        'system': 'System'
    };
    
    return nameMap[user] || user;
}

function getUserRole(user) {
    const roleMap = {
        'admin': 'Administrator',
        'staff': 'Staff Member',
        'system': 'System Process'
    };
    
    return roleMap[user] || user;
}

function getTypeBadge(type) {
    const typeMap = {
        'job': { text: 'Job', color: 'bg-blue-100 text-blue-800' },
        'customer': { text: 'Customer', color: 'bg-green-100 text-green-800' },
        'invoice': { text: 'Invoice', color: 'bg-purple-100 text-purple-800' },
        'system': { text: 'System', color: 'bg-gray-100 text-gray-800' }
    };
    
    const typeInfo = typeMap[type] || { text: type, color: 'bg-gray-100 text-gray-800' };
    
    return `<span class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${typeInfo.color}">${typeInfo.text}</span>`;
}

function getActivityColor(color) {
    const colorMap = {
        'green': '#10b981',
        'blue': '#3b82f6',
        'orange': '#f59e0b',
        'purple': '#8b5cf6',
        'yellow': '#eab308',
        'red': '#ef4444',
        'gray': '#6b7280'
    };
    
    return colorMap[color] || '#6b7280';
}

function formatActivityTime(timestamp) {
    const date = new Date(timestamp);
    return date.toLocaleString('en-GB', {
        day: '2-digit',
        month: 'short',
        hour: '2-digit',
        minute: '2-digit'
    });
}

function showActivityDetails(activityId) {
    // This would show detailed information about the activity
    showNotification('Activity details feature coming soon', 'info');
}

// Invoices Page Functions
function loadInvoicesTable() {
    console.log('Loading invoices table...');
    $.ajax({
        url: 'data/invoices.json?v=' + Date.now(),
        method: 'GET',
        dataType: 'json',
        success: function(data) {
            console.log('Invoices loaded successfully:', data.invoices.length, 'invoices');
            displayInvoicesTable(data.invoices);
            setupInvoiceFilters();
        },
        error: function(xhr, status, error) {
            console.log('Error loading invoices:', error);
            // Fallback to demo data
            const demoInvoices = [
                {
                    id: "INV-2024-001",
                    jobId: 1,
                    customer: "Direct Line Insurance",
                    description: "BMW 18\" Alloy Wheel Refurbishment",
                    regNo: "AB12 CDE",
                    amount: 480.00,
                    status: "paid",
                    issueDate: "2024-01-15",
                    dueDate: "2024-01-29"
                },
                {
                    id: "INV-2024-002",
                    jobId: 2,
                    customer: "Aviva Insurance",
                    description: "Mercedes 19\" Diamond Cut Wheels",
                    regNo: "XY98 ZAB",
                    amount: 600.00,
                    status: "pending",
                    issueDate: "2024-01-17",
                    dueDate: "2024-01-31"
                }
            ];
            console.log('Using demo data:', demoInvoices.length, 'invoices');
            displayInvoicesTable(demoInvoices);
            setupInvoiceFilters();
        }
    });
}

function displayInvoicesTable(invoices) {
    const tableBody = $('#invoices-table-body');
    console.log('Table body element:', tableBody.length);
    tableBody.empty();
    
    console.log('Displaying invoices:', invoices.length, 'invoices');
    
    // Sort invoices by status: overdue, pending, paid
    const sortedInvoices = sortInvoicesByStatus(invoices);
    console.log('Invoices sorted by status:', sortedInvoices.map(invoice => ({ id: invoice.id, status: invoice.status })));
    
    sortedInvoices.forEach((invoice, index) => {
        console.log(`Processing invoice ${index + 1}:`, invoice.id, invoice.customer, invoice.status);
        const row = createInvoiceTableRow(invoice);
        tableBody.append(row);
    });
    
    console.log('Total rows in table:', tableBody.find('tr').length);
}

function sortInvoicesByStatus(invoices) {
    const statusOrder = {
        'overdue': 1,
        'pending': 2,
        'paid': 3
    };
    
    return invoices.sort((a, b) => {
        const statusA = statusOrder[a.status] || 999;
        const statusB = statusOrder[b.status] || 999;
        
        // If same status, sort by due date (earliest first)
        if (statusA === statusB) {
            return new Date(a.dueDate) - new Date(b.dueDate);
        }
        
        return statusA - statusB;
    });
}

function createInvoiceTableRow(invoice) {
    console.log('Creating row for invoice:', invoice.id, invoice.customer);
    const statusClass = getInvoiceStatusClass(invoice.status);
    const statusText = getInvoiceStatusText(invoice.status);
    const isOverdue = isInvoiceOverdue(invoice);
    
    return $(`
        <tr class="table-row ${isOverdue ? 'bg-red-50' : ''}">
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">${invoice.id}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${invoice.customer}</td>
            <td class="px-6 py-4 whitespace-nowrap">
                <div class="text-sm text-gray-900">${invoice.description}</div>
                ${invoice.regNo ? `<div class="text-xs text-gray-500">${invoice.regNo}</div>` : ''}
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">£${invoice.amount.toFixed(2)}</td>
            <td class="px-6 py-4 whitespace-nowrap">
                <span class="invoice-status-badge ${statusClass}">${statusText}</span>
            </td>
            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">${formatDate(invoice.issueDate)}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm ${isOverdue ? 'text-red-600 font-medium' : 'text-gray-500'}">${formatDate(invoice.dueDate)}</td>
            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                <div class="flex items-center space-x-2">
                    <button onclick="viewInvoice('${invoice.id}')" class="text-blue-600 hover:text-blue-900" title="View Invoice">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button onclick="downloadInvoice('${invoice.id}')" class="text-green-600 hover:text-green-900" title="Download PDF">
                        <i class="fas fa-download"></i>
                    </button>
                    ${invoice.status === 'pending' ? 
                        `<button onclick="markAsPaid('${invoice.id}')" class="text-purple-600 hover:text-purple-900" title="Mark as Paid">
                            <i class="fas fa-check"></i>
                        </button>` : ''
                    }
                </div>
            </td>
        </tr>
    `);
}

function setupInvoiceFilters() {
    // Invoice status filter
    $('#invoice-status-filter').on('change', function() {
        filterInvoices();
    });
    
    // Customer filter
    $('#invoice-customer-filter').on('change', function() {
        filterInvoices();
    });
    
    // Date filter
    $('#invoice-date-from').on('change', function() {
        filterInvoices();
    });
    
    // Filter button
    $('#filter-invoices').on('click', function() {
        filterInvoices();
    });
}

function filterInvoices() {
    const statusFilter = $('#invoice-status-filter').val();
    const customerFilter = $('#invoice-customer-filter').val();
    const dateFilter = $('#invoice-date-from').val();
    
    // This would implement filtering logic
    // For now, just reload all invoices
    loadInvoicesTable();
}

function getInvoiceStatusClass(status) {
    const statusMap = {
        'pending': 'invoice-status-pending',
        'paid': 'invoice-status-paid',
        'overdue': 'invoice-status-overdue'
    };
    return statusMap[status] || 'invoice-status-pending';
}

function getInvoiceStatusText(status) {
    const statusMap = {
        'pending': 'Pending',
        'paid': 'Paid',
        'overdue': 'Overdue'
    };
    return statusMap[status] || 'Pending';
}

function isInvoiceOverdue(invoice) {
    if (invoice.status === 'paid') return false;
    const today = new Date();
    const dueDate = new Date(invoice.dueDate);
    return dueDate < today;
}

function viewInvoice(invoiceId) {
    showNotification(`Viewing invoice ${invoiceId}`, 'info');
    // This would open an invoice view modal
}

function downloadInvoice(invoiceId) {
    showNotification(`Downloading invoice ${invoiceId}`, 'info');
    // This would trigger PDF download
}

function markAsPaid(invoiceId) {
    showNotification(`Invoice ${invoiceId} marked as paid`, 'success');
    // This would update the invoice status
}

// Reports Page Functions
let charts = {}; // Store chart instances

function loadReportsPage() {
    console.log('Loading reports page...');
    
    // Initialize metrics
    updateMetrics();
    
    // Initialize all charts
    setTimeout(() => {
        initRevenueChart();
        initJobStatusChart();
        initServiceRevenueChart();
        initSatisfactionChart();
        initPaymentStatusChart();
        initMonthlyPerformanceChart();
        initTopCustomersChart();
        setupReportFilters();
    }, 100);
}

function updateMetrics() {
    // Simulate data - in real app, this would come from API
    const metrics = {
        totalJobsProcessed: 342,
        jobsCompleted: 127,
        activeCustomers: 23,
        avgJobValue: 360.00,
        jobsProcessedChange: 15.8,
        jobsChange: 8.3,
        customersChange: 15.2,
        avgValueChange: 5.7
    };
    
    $('#total-jobs-processed').text(metrics.totalJobsProcessed.toLocaleString());
    $('#jobs-completed').text(metrics.jobsCompleted);
    $('#active-customers').text(metrics.activeCustomers);
    $('#avg-job-value').text('£' + metrics.avgJobValue.toFixed(0));
    
    $('#jobs-processed-change').text('+' + metrics.jobsProcessedChange + '%');
    $('#jobs-change').text('+' + metrics.jobsChange + '%');
    $('#customers-change').text('+' + metrics.customersChange + '%');
    $('#avg-value-change').text('+' + metrics.avgValueChange + '%');
}

function initRevenueChart() {
    const chartDom = document.getElementById('revenue-chart');
    if (!chartDom) return;
    
    charts.revenue = echarts.init(chartDom);
    
    const option = {
        tooltip: {
            trigger: 'axis',
            axisPointer: {
                type: 'cross',
                label: {
                    backgroundColor: '#6a7985'
                }
            }
        },
        legend: {
            data: ['Revenue', 'Jobs Completed']
        },
        grid: {
            left: '3%',
            right: '4%',
            bottom: '3%',
            containLabel: true
        },
        xAxis: {
            type: 'category',
            boundaryGap: false,
            data: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun']
        },
        yAxis: [
            {
                type: 'value',
                name: 'Revenue (£)',
                position: 'left'
            },
            {
                type: 'value',
                name: 'Jobs',
                position: 'right'
            }
        ],
        series: [
            {
                name: 'Revenue',
                type: 'line',
                smooth: true,
                data: [3200, 4100, 3800, 5200, 4800, 3600, 2800],
                itemStyle: {
                    color: '#00a9ea'
                },
                areaStyle: {
                    color: {
                        type: 'linear',
                        x: 0, y: 0, x2: 0, y2: 1,
                        colorStops: [
                            { offset: 0, color: 'rgba(0, 169, 234, 0.3)' },
                            { offset: 1, color: 'rgba(0, 169, 234, 0.05)' }
                        ]
                    }
                }
            },
            {
                name: 'Jobs Completed',
                type: 'bar',
                yAxisIndex: 1,
                data: [8, 12, 10, 15, 14, 9, 6],
                itemStyle: {
                    color: '#10b981'
                }
            }
        ]
    };
    
    charts.revenue.setOption(option);
}

function initJobStatusChart() {
    const chartDom = document.getElementById('job-status-chart');
    if (!chartDom) return;
    
    charts.jobStatus = echarts.init(chartDom);
    
    const option = {
        tooltip: {
            trigger: 'item',
            formatter: '{a} <br/>{b}: {c} ({d}%)'
        },
        legend: {
            orient: 'vertical',
            left: 'left',
            data: ['Pending', 'In Progress', 'Ready', 'Completed', 'Invoiced']
        },
        series: [
            {
                name: 'Job Status',
                type: 'pie',
                radius: ['40%', '70%'],
                center: ['60%', '50%'],
                avoidLabelOverlap: false,
                itemStyle: {
                    borderRadius: 10,
                    borderColor: '#fff',
                    borderWidth: 2
                },
                label: {
                    show: false,
                    position: 'center'
                },
                emphasis: {
                    label: {
                        show: true,
                        fontSize: '18',
                        fontWeight: 'bold'
                    }
                },
                labelLine: {
                    show: false
                },
                data: [
                    { value: 15, name: 'Pending', itemStyle: { color: '#f59e0b' } },
                    { value: 28, name: 'In Progress', itemStyle: { color: '#3b82f6' } },
                    { value: 12, name: 'Ready', itemStyle: { color: '#8b5cf6' } },
                    { value: 35, name: 'Completed', itemStyle: { color: '#10b981' } },
                    { value: 20, name: 'Invoiced', itemStyle: { color: '#6366f1' } }
                ]
            }
        ]
    };
    
    charts.jobStatus.setOption(option);
}

function initServiceRevenueChart() {
    const chartDom = document.getElementById('service-revenue-chart');
    if (!chartDom) return;
    
    charts.serviceRevenue = echarts.init(chartDom);
    
    const option = {
        tooltip: {
            trigger: 'axis',
            axisPointer: {
                type: 'shadow'
            }
        },
        grid: {
            left: '3%',
            right: '4%',
            bottom: '3%',
            containLabel: true
        },
        xAxis: {
            type: 'category',
            data: ['Powder Coating', 'Diamond Cutting', 'Chrome Restoration', 'Standard Refurb', 'Other Services']
        },
        yAxis: {
            type: 'value',
            name: 'Revenue (£)'
        },
        series: [
            {
                name: 'Revenue',
                type: 'bar',
                data: [18500, 15200, 8900, 2100, 980],
                itemStyle: {
                    color: function(params) {
                        const colors = ['#00a9ea', '#10b981', '#f59e0b', '#ef4444', '#8b5cf6'];
                        return colors[params.dataIndex];
                    },
                    borderRadius: [4, 4, 0, 0]
                },
                emphasis: {
                    itemStyle: {
                        shadowBlur: 10,
                        shadowOffsetX: 0,
                        shadowColor: 'rgba(0, 0, 0, 0.5)'
                    }
                }
            }
        ]
    };
    
    charts.serviceRevenue.setOption(option);
}

function initSatisfactionChart() {
    const chartDom = document.getElementById('satisfaction-chart');
    if (!chartDom) return;
    
    charts.satisfaction = echarts.init(chartDom);
    
    const option = {
        tooltip: {
            trigger: 'item'
        },
        radar: {
            indicator: [
                { name: 'Quality', max: 100 },
                { name: 'Speed', max: 100 },
                { name: 'Communication', max: 100 },
                { name: 'Value', max: 100 },
                { name: 'Overall', max: 100 }
            ],
            radius: '70%',
            axisName: {
                color: '#666'
            }
        },
        series: [
            {
                name: 'Customer Satisfaction',
                type: 'radar',
                data: [
                    {
                        value: [95, 88, 92, 85, 90],
                        name: 'Current Period',
                        itemStyle: {
                            color: '#00a9ea'
                        },
                        areaStyle: {
                            color: 'rgba(0, 169, 234, 0.2)'
                        }
                    },
                    {
                        value: [90, 85, 88, 80, 87],
                        name: 'Previous Period',
                        itemStyle: {
                            color: '#10b981'
                        },
                        areaStyle: {
                            color: 'rgba(16, 185, 129, 0.1)'
                        }
                    }
                ]
            }
        ]
    };
    
    charts.satisfaction.setOption(option);
}

function initPaymentStatusChart() {
    const chartDom = document.getElementById('payment-status-chart');
    if (!chartDom) return;
    
    charts.paymentStatus = echarts.init(chartDom);
    
    const option = {
        tooltip: {
            trigger: 'item'
        },
        series: [
            {
                name: 'Payment Status',
                type: 'pie',
                radius: '70%',
                data: [
                    { value: 65, name: 'Paid', itemStyle: { color: '#10b981' } },
                    { value: 25, name: 'Pending', itemStyle: { color: '#f59e0b' } },
                    { value: 10, name: 'Overdue', itemStyle: { color: '#ef4444' } }
                ],
                emphasis: {
                    itemStyle: {
                        shadowBlur: 10,
                        shadowOffsetX: 0,
                        shadowColor: 'rgba(0, 0, 0, 0.5)'
                    }
                }
            }
        ]
    };
    
    charts.paymentStatus.setOption(option);
}

function initMonthlyPerformanceChart() {
    const chartDom = document.getElementById('monthly-performance-chart');
    if (!chartDom) return;
    
    charts.monthlyPerformance = echarts.init(chartDom);
    
    const option = {
        tooltip: {
            trigger: 'axis',
            axisPointer: {
                type: 'cross'
            }
        },
        legend: {
            data: ['Jobs', 'Revenue (K)']
        },
        xAxis: {
            type: 'category',
            data: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun']
        },
        yAxis: [
            {
                type: 'value',
                name: 'Jobs',
                position: 'left'
            },
            {
                type: 'value',
                name: 'Revenue (K)',
                position: 'right'
            }
        ],
        series: [
            {
                name: 'Jobs',
                type: 'bar',
                data: [45, 52, 48, 61, 55, 67],
                itemStyle: {
                    color: '#3b82f6'
                }
            },
            {
                name: 'Revenue (K)',
                type: 'line',
                yAxisIndex: 1,
                data: [16.2, 18.7, 17.3, 22.1, 19.8, 24.1],
                itemStyle: {
                    color: '#10b981'
                },
                smooth: true
            }
        ]
    };
    
    charts.monthlyPerformance.setOption(option);
}

function initTopCustomersChart() {
    const chartDom = document.getElementById('top-customers-chart');
    if (!chartDom) return;
    
    charts.topCustomers = echarts.init(chartDom);
    
    const option = {
        tooltip: {
            trigger: 'axis',
            axisPointer: {
                type: 'shadow'
            }
        },
        grid: {
            left: '3%',
            right: '4%',
            bottom: '3%',
            containLabel: true
        },
        xAxis: {
            type: 'value',
            name: 'Revenue (£)'
        },
        yAxis: {
            type: 'category',
            data: ['Direct Line', 'Aviva', 'Admiral', 'Churchill', 'LV=']
        },
        series: [
            {
                name: 'Revenue',
                type: 'bar',
                data: [8500, 7200, 6800, 5200, 4800],
                itemStyle: {
                    color: function(params) {
                        const colors = ['#00a9ea', '#10b981', '#f59e0b', '#ef4444', '#8b5cf6'];
                        return colors[params.dataIndex];
                    }
                }
            }
        ]
    };
    
    charts.topCustomers.setOption(option);
}

function setupReportFilters() {
    $('#generate-report').on('click', function() {
        const period = $('#report-period').val();
        const reportType = $('#report-type').val();
        const chartType = $('#chart-type').val();
        
        showNotification(`Generating ${reportType} report for ${period}...`, 'info');
        
        // Simulate report generation
        setTimeout(() => {
            updateChartsForPeriod(period, reportType, chartType);
            showNotification('Report generated successfully!', 'success');
        }, 1000);
    });
    
    // Chart type buttons
    $('.bg-white .flex.space-x-2 button').on('click', function() {
        $(this).removeClass('bg-gray-100 text-gray-600').addClass('bg-blue-100 text-blue-600');
        $(this).siblings().removeClass('bg-blue-100 text-blue-600').addClass('bg-gray-100 text-gray-600');
    });
}

function updateChartsForPeriod(period, reportType, chartType) {
    // This would update charts based on selected filters
    console.log('Updating charts for:', period, reportType, chartType);
    
    // Resize all charts
    Object.values(charts).forEach(chart => {
        if (chart && chart.resize) {
            chart.resize();
        }
    });
}

// Handle window resize
$(window).on('resize', function() {
    Object.values(charts).forEach(chart => {
        if (chart && chart.resize) {
            chart.resize();
        }
    });
});

// Global Search Functionality
function initializeGlobalSearch() {
    // Load all data for global search
    loadGlobalSearchData();
    
    // Setup search input handler
    $('#global-search').on('input', function() {
        const searchTerm = $(this).val().toLowerCase().trim();
        
        if (searchTerm.length < 2) {
            $('#search-results').addClass('hidden');
            return;
        }
        
        const suggestions = generateSearchSuggestions(searchTerm);
        displaySearchSuggestions(suggestions);
    });
    
    // Hide search results when clicking outside
    $(document).on('click', function(e) {
        if (!$(e.target).closest('#global-search, #search-results').length) {
            $('#search-results').addClass('hidden');
        }
    });
    
    // Handle search result clicks
    $(document).on('click', '.search-suggestion', function() {
        const type = $(this).data('type');
        const id = $(this).data('id');
        handleSearchResultClick(type, id);
    });
}

function loadGlobalSearchData() {
    // Load jobs data
    $.ajax({
        url: 'data/jobs.json?v=' + Date.now(),
        method: 'GET',
        dataType: 'json',
        success: function(data) {
            globalSearchData.jobs = data.jobs;
        }
    });
    
    // Load customers data (mock data for now)
    globalSearchData.customers = [
        { id: 1, name: "Direct Line Insurance", type: "Insurance" },
        { id: 2, name: "Aviva Insurance", type: "Insurance" },
        { id: 3, name: "Admiral Insurance", type: "Insurance" },
        { id: 4, name: "Churchill Insurance", type: "Insurance" },
        { id: 5, name: "LV= Insurance", type: "Insurance" },
        { id: 6, name: "Hastings Direct", type: "Insurance" },
        { id: 7, name: "More Than Insurance", type: "Insurance" },
        { id: 8, name: "Tesco Bank Insurance", type: "Insurance" },
        { id: 9, name: "Esure Insurance", type: "Insurance" },
        { id: 10, name: "Co-op Insurance", type: "Insurance" }
    ];
    
    // Load invoices data (mock data for now)
    globalSearchData.invoices = [
        { id: "INV-2024-001", customer: "Direct Line Insurance", amount: 480.00, status: "paid" },
        { id: "INV-2024-002", customer: "Aviva Insurance", amount: 600.00, status: "pending" },
        { id: "INV-2024-003", customer: "Admiral Insurance", amount: 280.00, status: "paid" },
        { id: "INV-2024-004", customer: "Churchill Insurance", amount: 240.00, status: "overdue" },
        { id: "INV-2024-005", customer: "LV= Insurance", amount: 780.00, status: "pending" }
    ];
}

function generateSearchSuggestions(searchTerm) {
    const suggestions = [];
    
    // Search jobs
    globalSearchData.jobs.forEach(job => {
        if (job.customer.toLowerCase().includes(searchTerm) ||
            job.description.toLowerCase().includes(searchTerm) ||
            (job.regNo && job.regNo.toLowerCase().includes(searchTerm)) ||
            (job.service && job.service.toLowerCase().includes(searchTerm))) {
            
            suggestions.push({
                type: 'job',
                id: job.id,
                title: job.description,
                subtitle: `${job.customer} • ${job.regNo || 'No Reg'}`,
                icon: 'fas fa-cog',
                color: 'text-blue-600'
            });
        }
    });
    
    // Search customers
    globalSearchData.customers.forEach(customer => {
        if (customer.name.toLowerCase().includes(searchTerm)) {
            suggestions.push({
                type: 'customer',
                id: customer.id,
                title: customer.name,
                subtitle: customer.type,
                icon: 'fas fa-building',
                color: 'text-green-600'
            });
        }
    });
    
    // Search invoices
    globalSearchData.invoices.forEach(invoice => {
        if (invoice.id.toLowerCase().includes(searchTerm) ||
            invoice.customer.toLowerCase().includes(searchTerm)) {
            
            suggestions.push({
                type: 'invoice',
                id: invoice.id,
                title: invoice.id,
                subtitle: `${invoice.customer} • £${invoice.amount.toFixed(2)}`,
                icon: 'fas fa-file-invoice',
                color: 'text-purple-600'
            });
        }
    });
    
    // Add keyword-based suggestions
    addKeywordSuggestions(searchTerm, suggestions);
    
    return suggestions.slice(0, 8); // Limit to 8 suggestions
}

function addKeywordSuggestions(searchTerm, suggestions) {
    const keywordMap = {
        'reg': { type: 'job', field: 'regNo', icon: 'fas fa-car', color: 'text-yellow-600' },
        'customer': { type: 'customer', field: 'name', icon: 'fas fa-building', color: 'text-green-600' },
        'invoice': { type: 'invoice', field: 'id', icon: 'fas fa-file-invoice', color: 'text-purple-600' },
        'job': { type: 'job', field: 'description', icon: 'fas fa-cog', color: 'text-blue-600' },
        'pending': { type: 'job', field: 'status', value: 'pending', icon: 'fas fa-clock', color: 'text-amber-600' },
        'progress': { type: 'job', field: 'status', value: 'in-progress', icon: 'fas fa-tools', color: 'text-blue-600' },
        'ready': { type: 'job', field: 'status', value: 'ready', icon: 'fas fa-check-circle', color: 'text-green-600' },
        'completed': { type: 'job', field: 'status', value: 'completed', icon: 'fas fa-check', color: 'text-green-600' },
        'paid': { type: 'invoice', field: 'status', value: 'paid', icon: 'fas fa-check-circle', color: 'text-green-600' },
        'overdue': { type: 'invoice', field: 'status', value: 'overdue', icon: 'fas fa-exclamation-triangle', color: 'text-red-600' }
    };
    
    Object.keys(keywordMap).forEach(keyword => {
        if (searchTerm.includes(keyword)) {
            const config = keywordMap[keyword];
            suggestions.unshift({
                type: 'suggestion',
                id: keyword,
                title: `Search by ${keyword}`,
                subtitle: `Filter ${config.type}s by ${keyword}`,
                icon: config.icon,
                color: config.color,
                isKeyword: true
            });
        }
    });
}

function displaySearchSuggestions(suggestions) {
    const container = $('#search-suggestions');
    container.empty();
    
    if (suggestions.length === 0) {
        container.html('<div class="p-3 text-center text-gray-500">No results found</div>');
    } else {
        suggestions.forEach(suggestion => {
            const suggestionHtml = `
                <div class="search-suggestion flex items-center p-3 hover:bg-gray-50 rounded-lg cursor-pointer transition-colors" 
                     data-type="${suggestion.type}" data-id="${suggestion.id}">
                    <div class="w-8 h-8 bg-gray-100 rounded-lg flex items-center justify-center mr-3">
                        <i class="${suggestion.icon} ${suggestion.color} text-sm"></i>
                    </div>
                    <div class="flex-1">
                        <div class="font-medium text-gray-900">${suggestion.title}</div>
                        <div class="text-sm text-gray-500">${suggestion.subtitle}</div>
                    </div>
                    ${suggestion.isKeyword ? '<i class="fas fa-arrow-right text-gray-400"></i>' : ''}
                </div>
            `;
            container.append(suggestionHtml);
        });
    }
    
    $('#search-results').removeClass('hidden');
}

function handleSearchResultClick(type, id) {
    $('#search-results').addClass('hidden');
    $('#global-search').val('');
    
    switch(type) {
        case 'job':
            showJobDetails(id);
            break;
        case 'customer':
            // Navigate to customers page and highlight customer
            showPage('customers');
            break;
        case 'invoice':
            // Navigate to invoices page and highlight invoice
            showPage('invoices');
            break;
        case 'suggestion':
            handleKeywordSuggestion(id);
            break;
    }
}

function handleKeywordSuggestion(keyword) {
    // Implement keyword-based filtering
    switch(keyword) {
        case 'reg':
            $('#global-search').val('reg:').focus();
            break;
        case 'customer':
            $('#global-search').val('customer:').focus();
            break;
        case 'invoice':
            $('#global-search').val('invoice:').focus();
            break;
        case 'pending':
            showPage('jobs');
            // Could add status filter here
            break;
        case 'progress':
            showPage('jobs');
            // Could add status filter here
            break;
        case 'ready':
            showPage('jobs');
            // Could add status filter here
            break;
    }
}

function logout() {
    localStorage.removeItem('staff');
    showNotification('Logged out successfully', 'info');
    
    // In a real app, redirect to login page
    setTimeout(() => {
        window.location.href = 'login.html';
    }, 1000);
}
