<?php
// Search for a specific invoice by ID
echo "=== Searching for Specific Invoice ===\n\n";

// Load stored tokens
$tokensFile = 'xero_tokens.json';
if (!file_exists($tokensFile)) {
    echo "❌ No tokens found. Please authorize the app first.\n";
    exit;
}

$tokens = json_decode(file_get_contents($tokensFile), true);
if (!$tokens || !isset($tokens['access_token'])) {
    echo "❌ Invalid token file. Please re-authorize the app.\n";
    exit;
}

$accessToken = $tokens['access_token'];
$tenantId = $tokens['tenant_id'];

// The invoice ID we're looking for (updated with latest)
$targetInvoiceId = '74d5da48-a880-4b4f-8169-4d300fb05475';
$targetInvoiceNumber = 'INV-0046';

echo "1. Searching for invoice:\n";
echo "   Invoice ID: $targetInvoiceId\n";
echo "   Invoice Number: $targetInvoiceNumber\n\n";

echo "2. Fetching all invoices from Xero...\n";

// Call Xero's invoices endpoint
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'https://api.xero.com/api.xro/2.0/Invoices');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Authorization: Bearer ' . $accessToken,
    'Accept: application/json',
    'Xero-tenant-id: ' . $tenantId
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

echo "3. API Response:\n";
echo "   HTTP Code: $httpCode\n";
if ($error) {
    echo "   Error: $error\n";
    exit;
}

// Parse the response
$data = json_decode($response, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    echo "❌ Failed to parse JSON response: " . json_last_error_msg() . "\n";
    exit;
}

if (isset($data['Invoices'])) {
    $invoices = $data['Invoices'];
    echo "   ✅ Found " . count($invoices) . " invoice(s) in Xero\n\n";
    
    // Search for our specific invoice
    $foundById = false;
    $foundByNumber = false;
    $foundInvoice = null;
    
    foreach ($invoices as $invoice) {
        $invoiceId = $invoice['InvoiceID'] ?? '';
        $invoiceNumber = $invoice['InvoiceNumber'] ?? '';
        
        if ($invoiceId === $targetInvoiceId) {
            $foundById = true;
            $foundInvoice = $invoice;
            break;
        }
        
        if ($invoiceNumber === $targetInvoiceNumber) {
            $foundByNumber = true;
            if (!$foundInvoice) {
                $foundInvoice = $invoice;
            }
        }
    }
    
    echo "4. Search Results:\n";
    echo "   Found by Invoice ID: " . ($foundById ? "✅ YES" : "❌ NO") . "\n";
    echo "   Found by Invoice Number: " . ($foundByNumber ? "✅ YES" : "❌ NO") . "\n\n";
    
    if ($foundInvoice) {
        echo "5. Invoice Details:\n";
        echo "   ├─ Invoice ID: " . ($foundInvoice['InvoiceID'] ?? 'Unknown') . "\n";
        echo "   ├─ Invoice Number: " . ($foundInvoice['InvoiceNumber'] ?? 'Unknown') . "\n";
        echo "   ├─ Contact: " . ($foundInvoice['Contact']['Name'] ?? 'Unknown') . "\n";
        echo "   ├─ Total: " . ($foundInvoice['Total'] ?? 'Unknown') . "\n";
        echo "   ├─ Status: " . ($foundInvoice['Status'] ?? 'Unknown') . "\n";
        echo "   ├─ Date: " . ($foundInvoice['Date'] ?? 'Unknown') . "\n";
        echo "   └─ Due Date: " . ($foundInvoice['DueDate'] ?? 'Unknown') . "\n\n";
        
        echo "✅ SUCCESS: The invoice exists in your Xero account!\n";
        echo "   You should be able to see it in your Xero dashboard.\n";
    } else {
        echo "❌ INVOICE NOT FOUND\n";
        echo "   This could mean:\n";
        echo "   - The invoice creation failed silently\n";
        echo "   - There's a delay in Xero's API\n";
        echo "   - The invoice was created in a different organization\n";
        echo "   - There's a permission issue\n\n";
        
        echo "6. Recent invoices (last 5) for reference:\n";
        $recentInvoices = array_slice($invoices, 0, 5);
        foreach ($recentInvoices as $index => $invoice) {
            echo "   Invoice #" . ($index + 1) . ": " . ($invoice['InvoiceNumber'] ?? 'Unknown') . 
                 " - " . ($invoice['Contact']['Name'] ?? 'Unknown') . 
                 " - " . ($invoice['Total'] ?? 'Unknown') . "\n";
        }
    }
    
} else {
    echo "   ❌ Unexpected response format\n";
    echo "   Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
}

echo "\n=== Search Complete ===\n";
?>
