<?php
// Test contact creation with full details
echo "=== Testing Contact Creation ===\n\n";

require_once 'vendor/autoload.php';

use XeroAPI\XeroPHP\Api\AccountingApi;
use XeroAPI\XeroPHP\Configuration;
use XeroAPI\XeroPHP\Models\Accounting\Contact;
use XeroAPI\XeroPHP\Models\Accounting\Address;
use XeroAPI\XeroPHP\Models\Accounting\Phone;
use XeroAPI\XeroPHP\Models\Accounting\Contacts;
use GuzzleHttp\Client;

// Load tokens
$tokens = json_decode(file_get_contents('xero_tokens.json'), true);
$accessToken = $tokens['access_token'];
$tenantId = $tokens['tenant_id'];

// Configure Xero API
$config = Configuration::getDefaultConfiguration();
$config->setAccessToken($accessToken);
$config->setHost('https://api.xero.com');

$guzzleClient = new Client([
    'verify' => false,
    'timeout' => 30
]);

$apiInstance = new AccountingApi($guzzleClient, $config);

try {
    // Create a test contact with full details
    $contact = new Contact();
    $contact->setName('Test Insurance Company Ltd');
    
    // Add address
    $address = new Address();
    $address->setAddressType('STREET');
    $address->setAddressLine1('123 Test Street');
    $address->setCity('London');
    $address->setPostalCode('SW1A 1AA');
    $address->setCountry('United Kingdom');
    $contact->setAddresses([$address]);
    
    // Add phone
    $contact->setPhones([
        (new Phone())
            ->setPhoneType('MOBILE')
            ->setPhoneNumber('0800 123 456')
    ]);
    
    // Add email
    $contact->setEmailAddress('test@insurance.com');
    $contact->setIsCustomer(true);
    $contact->setContactStatus('ACTIVE');
    
    echo "1. Creating contact with details:\n";
    echo "   Name: " . $contact->getName() . "\n";
    echo "   Email: " . $contact->getEmailAddress() . "\n";
    echo "   Phone: " . $contact->getPhones()[0]->getPhoneNumber() . "\n";
    echo "   Address: " . $contact->getAddresses()[0]->getAddressLine1() . "\n";
    echo "   City: " . $contact->getAddresses()[0]->getCity() . "\n";
    echo "   Postcode: " . $contact->getAddresses()[0]->getPostalCode() . "\n\n";
    
    // Create contacts wrapper
    $contacts = new Contacts();
    $contacts->setContacts([$contact]);
    
    echo "2. Sending to Xero API...\n";
    
    $result = $apiInstance->createContacts($tenantId, $contacts);
    
    echo "3. API Response:\n";
    if ($result->getContacts() && count($result->getContacts()) > 0) {
        $createdContact = $result->getContacts()[0];
        echo "   ✅ SUCCESS: Contact created!\n";
        echo "   Contact ID: " . $createdContact->getContactID() . "\n";
        echo "   Name: " . $createdContact->getName() . "\n";
        echo "   Email: " . ($createdContact->getEmailAddress() ?? 'Not set') . "\n";
        echo "   Phone: " . (count($createdContact->getPhones()) > 0 ? $createdContact->getPhones()[0]->getPhoneNumber() : 'Not set') . "\n";
        
        $addresses = $createdContact->getAddresses();
        if (!empty($addresses)) {
            $addr = $addresses[0];
            echo "   Address: " . $addr->getAddressLine1() . "\n";
            echo "   City: " . $addr->getCity() . "\n";
            echo "   Postcode: " . $addr->getPostalCode() . "\n";
            echo "   Country: " . $addr->getCountry() . "\n";
        } else {
            echo "   Address: Not set\n";
        }
    } else {
        echo "   ❌ No contacts returned\n";
    }
    
} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
    echo "   File: " . $e->getFile() . "\n";
    echo "   Line: " . $e->getLine() . "\n";
}

echo "\n=== Test Complete ===\n";
?>
