<?php
// Verify that contacts are created with full details in Xero
echo "=== Verifying Contact Details in Xero ===\n\n";

require_once 'vendor/autoload.php';

use XeroAPI\XeroPHP\Api\AccountingApi;
use XeroAPI\XeroPHP\Configuration;
use GuzzleHttp\Client;

// Load tokens
$tokens = json_decode(file_get_contents('xero_tokens.json'), true);
$accessToken = $tokens['access_token'];
$tenantId = $tokens['tenant_id'];

// Configure Xero API
$config = Configuration::getDefaultConfiguration();
$config->setAccessToken($accessToken);
$config->setHost('https://api.xero.com');

$guzzleClient = new Client([
    'verify' => false,
    'timeout' => 30
]);

$apiInstance = new AccountingApi($guzzleClient, $config);

try {
    // Get all contacts
    $contacts = $apiInstance->getContacts($tenantId);
    
    echo "1. Found " . count($contacts->getContacts()) . " contacts in Xero\n\n";
    
    // Look for recent insurance company contacts
    $insuranceCompanies = [
        'Churchill Insurance',
        'Green Flag Insurance', 
        'Aviva Insurance Ltd',
        'Direct Line Insurance',
        'Admiral Insurance',
        'More Than Insurance',
        'LV= Insurance',
        'Hastings Direct',
        'Esure Insurance',
        'Tesco Bank Insurance',
        'Sainsbury\'s Bank Insurance',
        'Halifax Insurance',
        'Barclays Insurance',
        'AA Insurance',
        'RAC Insurance'
    ];
    
    $foundContacts = [];
    foreach ($contacts->getContacts() as $contact) {
        if (in_array($contact->getName(), $insuranceCompanies)) {
            $foundContacts[] = $contact;
        }
    }
    
    echo "2. Found " . count($foundContacts) . " insurance company contacts\n\n";
    
    // Show details of the most recent contact
    if (!empty($foundContacts)) {
        $recentContact = $foundContacts[0];
        echo "3. Most Recent Contact Details:\n";
        echo "   Name: " . $recentContact->getName() . "\n";
        echo "   Contact ID: " . $recentContact->getContactID() . "\n";
        echo "   Email: " . ($recentContact->getEmailAddress() ?? 'Not set') . "\n";
        echo "   Phone: " . (count($recentContact->getPhones()) > 0 ? $recentContact->getPhones()[0]->getPhoneNumber() : 'Not set') . "\n";
        
        // Check addresses
        $addresses = $recentContact->getAddresses();
        if (!empty($addresses)) {
            $address = $addresses[0];
            echo "   Address: " . $address->getAddressLine1() . "\n";
            echo "   City: " . $address->getCity() . "\n";
            echo "   Postcode: " . $address->getPostalCode() . "\n";
            echo "   Country: " . $address->getCountry() . "\n";
        } else {
            echo "   Address: Not set\n";
        }
        
        echo "   Is Customer: " . ($recentContact->getIsCustomer() ? 'Yes' : 'No') . "\n";
        echo "   Status: " . $recentContact->getContactStatus() . "\n";
        
    } else {
        echo "3. No insurance company contacts found\n";
    }
    
    echo "\n✅ Contact verification complete!\n";
    
} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
}

echo "\n=== Verification Complete ===\n";
?>
