<?php
// Verify if the invoice was actually created in Xero
echo "=== Verifying Invoice in Xero ===\n\n";

// Load stored tokens
$tokensFile = 'xero_tokens.json';
if (!file_exists($tokensFile)) {
    echo "❌ No tokens found. Please authorize the app first.\n";
    exit;
}

$tokens = json_decode(file_get_contents($tokensFile), true);
if (!$tokens || !isset($tokens['access_token'])) {
    echo "❌ Invalid token file. Please re-authorize the app.\n";
    exit;
}

$accessToken = $tokens['access_token'];
$tenantId = $tokens['tenant_id'];

echo "1. Using credentials:\n";
echo "   Access Token: " . substr($accessToken, 0, 50) . "...\n";
echo "   Tenant ID: $tenantId\n\n";

echo "2. Fetching invoices from Xero...\n";

// Call Xero's invoices endpoint to get all invoices
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, 'https://api.xero.com/api.xro/2.0/Invoices');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Authorization: Bearer ' . $accessToken,
    'Accept: application/json',
    'Xero-tenant-id: ' . $tenantId
]);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);
curl_close($ch);

echo "3. API Response:\n";
echo "   HTTP Code: $httpCode\n";
if ($error) {
    echo "   Error: $error\n";
    exit;
}

echo "   Raw Response Length: " . strlen($response) . " characters\n\n";

// Parse the response
$data = json_decode($response, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    echo "❌ Failed to parse JSON response: " . json_last_error_msg() . "\n";
    echo "Raw response: " . substr($response, 0, 500) . "...\n";
    exit;
}

echo "4. Parsed Response:\n";
if (isset($data['Invoices'])) {
    $invoices = $data['Invoices'];
    echo "   ✅ Found " . count($invoices) . " invoice(s) in Xero\n\n";
    
    if (empty($invoices)) {
        echo "   ❌ No invoices found. This could mean:\n";
        echo "   - The invoice wasn't actually created\n";
        echo " - You're looking in the wrong organization\n";
        echo "   - There's a permission issue\n\n";
    } else {
        echo "   Recent invoices:\n";
        foreach (array_slice($invoices, 0, 5) as $index => $invoice) {
            echo "   Invoice #" . ($index + 1) . ":\n";
            echo "   ├─ Invoice ID: " . ($invoice['InvoiceID'] ?? 'Not provided') . "\n";
            echo "   ├─ Invoice Number: " . ($invoice['InvoiceNumber'] ?? 'Not provided') . "\n";
            echo "   ├─ Status: " . ($invoice['Status'] ?? 'Not provided') . "\n";
            echo "   ├─ Date: " . ($invoice['Date'] ?? 'Not provided') . "\n";
            echo "   ├─ Due Date: " . ($invoice['DueDate'] ?? 'Not provided') . "\n";
            echo "   ├─ Total: " . ($invoice['Total'] ?? 'Not provided') . "\n";
            echo "   └─ Contact: " . ($invoice['Contact']['Name'] ?? 'Not provided') . "\n\n";
        }
        
        // Look for our test invoice
        $testInvoiceFound = false;
        foreach ($invoices as $invoice) {
            if (strpos($invoice['InvoiceNumber'] ?? '', 'AUTO-') === 0) {
                echo "   🎯 Found RefurbFlow invoice: " . ($invoice['InvoiceNumber'] ?? 'Unknown') . "\n";
                $testInvoiceFound = true;
                break;
            }
        }
        
        if (!$testInvoiceFound) {
            echo "   ❌ No RefurbFlow invoices found (AUTO-* pattern)\n";
        }
    }
} else {
    echo "   ❌ Unexpected response format\n";
    echo "   Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
}

echo "\n5. Troubleshooting Tips:\n";
echo "   - Check if you're logged into the correct Xero organization\n";
echo "   - Look for invoices with 'AUTO-' prefix in the invoice number\n";
echo "   - Check the date range - invoices might be in a different period\n";
echo "   - Verify you have permission to view invoices\n";
echo "   - Try refreshing your Xero dashboard\n\n";

echo "=== Verification Complete ===\n";
?>
