// Van Driver App JavaScript
class VanDriverApp {
    constructor() {
        this.currentTab = 'collections';
        this.jobs = {
            collections: [],
            deliveries: [],
            completed: []
        };
        this.currentJobId = null;
        this.camera = null;
        this.cameraStream = null;
        
        this.init();
    }

    async init() {
        this.setupEventListeners();
        await this.loadJobs();
        this.renderCurrentTab();
        this.updateBadges();
    }

    setupEventListeners() {
        // Navigation tabs
        document.querySelectorAll('.nav-tab').forEach(tab => {
            tab.addEventListener('click', (e) => {
                const tabName = e.currentTarget.dataset.tab;
                this.switchTab(tabName);
            });
        });

        // Modal close handlers
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('modal')) {
                this.closeAllModals();
            }
        });

        // Escape key to close modals
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                this.closeAllModals();
            }
            // Arrow key navigation for job details
            if (document.getElementById('job-modal').classList.contains('active')) {
                if (e.key === 'ArrowLeft' || e.key === 'Backspace') {
                    this.closeJobModal();
                }
            }
        });
    }

    async loadJobs() {
        try {
            // Force refresh with updated sample data (remove this line in production)
            // Initialize with sample data
            this.jobs = await this.generateSampleJobs();
            this.saveJobs();
        } catch (error) {
            console.error('Error loading jobs:', error);
            // Fallback to sample data
            this.jobs = await this.generateSampleJobs();
        }
    }

    async generateSampleJobs() {
        return {
            collections: [
                {
                    id: 'COL001',
                    title: 'BMW 18" Alloy Wheels - Collection',
                    customer: 'ABC Insurance Ltd',
                    regNo: 'BM21 ABC',
                    address: '123 High Street, Manchester, M1 1AA',
                    phone: '+44 161 123 4567',
                    service: 'Diamond Cut',
                    priority: 'high',
                    status: 'pending',
                    estimatedTime: '2 hours',
                    quantity: 4,
                    wheelSize: '18"',
                    notes: 'Customer prefers morning collection'
                },
                {
                    id: 'COL002',
                    title: 'Mercedes 19" AMG Wheels - Collection',
                    customer: 'Direct Line Insurance',
                    regNo: 'MB19 XYZ',
                    address: '456 Oak Avenue, Birmingham, B2 4DT',
                    phone: '+44 121 987 6543',
                    service: 'Powder Coat',
                    priority: 'medium',
                    status: 'pending',
                    estimatedTime: '1.5 hours',
                    quantity: 4,
                    wheelSize: '19"',
                    notes: 'Wheels stored in garage'
                },
                {
                    id: 'COL003',
                    title: 'Audi 17" Alloys - Collection',
                    customer: 'LV Insurance',
                    regNo: 'AU17 DEF',
                    address: '789 Pine Road, Leeds, LS1 3HG',
                    phone: '+44 113 456 7890',
                    service: 'Standard Refurb',
                    priority: 'low',
                    status: 'pending',
                    estimatedTime: '1 hour',
                    quantity: 4,
                    wheelSize: '17"',
                    notes: 'Ring doorbell twice'
                }
            ],
            deliveries: [
                {
                    id: 'DEL001',
                    title: 'Ford 16" Steel Wheels - Delivery',
                    customer: 'Aviva Insurance',
                    regNo: 'FD16 GHI',
                    address: '321 Elm Street, Liverpool, L1 8JF',
                    phone: '+44 151 234 5678',
                    service: 'Powder Coat',
                    priority: 'high',
                    status: 'in-transit',
                    estimatedTime: '45 minutes',
                    quantity: 4,
                    wheelSize: '16"',
                    notes: 'Customer works from home'
                },
                {
                    id: 'DEL002',
                    title: 'VW 18" Alloys - Delivery',
                    customer: 'Admiral Insurance',
                    regNo: 'VW18 JKL',
                    address: '654 Maple Drive, Sheffield, S1 2NB',
                    phone: '+44 114 345 6789',
                    service: 'Diamond Cut',
                    priority: 'medium',
                    status: 'in-transit',
                    estimatedTime: '1 hour',
                    quantity: 4,
                    wheelSize: '18"',
                    notes: 'Call before delivery'
                }
            ],
            completed: [
                {
                    id: 'COM001',
                    title: 'BMW X5 20" Wheels - Complete',
                    customer: 'Zurich Insurance',
                    regNo: 'BM20 MNO',
                    address: '987 Cedar Lane, Newcastle, NE1 4RT',
                    phone: '+44 191 456 7890',
                    service: 'Diamond Cut',
                    priority: 'high',
                    status: 'delivered',
                    completedAt: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000).toISOString(),
                    quantity: 4,
                    wheelSize: '20"',
                    notes: 'Perfect finish achieved'
                },
                {
                    id: 'COM002',
                    title: 'Mercedes C-Class 17" - Complete',
                    customer: 'AXA Insurance',
                    regNo: 'MB17 PQR',
                    address: '147 Birch Road, Bristol, BS1 6TY',
                    phone: '+44 117 567 8901',
                    service: 'Powder Coat',
                    priority: 'medium',
                    status: 'delivered',
                    completedAt: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000).toISOString(),
                    quantity: 4,
                    wheelSize: '17"',
                    notes: 'Customer very satisfied'
                },
                {
                    id: 'COM003',
                    title: 'Audi A4 18" Alloys - Complete',
                    customer: 'Churchill Insurance',
                    regNo: 'AU18 STU',
                    address: '258 Willow Way, Cardiff, CF1 3AB',
                    phone: '+44 29 678 9012',
                    service: 'Standard Refurb',
                    priority: 'low',
                    status: 'delivered',
                    completedAt: new Date(Date.now() - 3 * 24 * 60 * 60 * 1000).toISOString(),
                    quantity: 4,
                    wheelSize: '18"',
                    notes: 'Quick turnaround requested'
                }
            ]
        };
    }

    saveJobs() {
        localStorage.setItem('vanDriverJobs', JSON.stringify(this.jobs));
    }

    switchTab(tabName) {
        // Update active tab
        document.querySelectorAll('.nav-tab').forEach(tab => {
            tab.classList.remove('active');
        });
        document.querySelector(`[data-tab="${tabName}"]`).classList.add('active');

        // Update active content
        document.querySelectorAll('.tab-content').forEach(content => {
            content.classList.remove('active');
        });
        document.getElementById(tabName).classList.add('active');

        this.currentTab = tabName;
        this.renderCurrentTab();
    }

    renderCurrentTab() {
        const container = document.getElementById(`${this.currentTab}-grid`);
        const jobs = this.jobs[this.currentTab];

        if (jobs.length === 0) {
            container.innerHTML = this.renderEmptyState();
            return;
        }

        container.innerHTML = jobs.map(job => this.renderJobCard(job)).join('');
    }

    renderJobCard(job) {
        const priorityClass = `priority-${job.priority}`;
        const statusClass = `status-${job.status}`;
        
        let actionButtons = '';
        
        if (this.currentTab === 'collections') {
            actionButtons = `
                <button class="btn btn-success" onclick="app.markAsCollected('${job.id}')">
                    <i class="fas fa-check"></i>
                    Mark Collected
                </button>
                <button class="btn btn-primary" onclick="app.capturePhoto('${job.id}', 'before')">
                    <i class="fas fa-camera"></i>
                    Take Photo
                </button>
                <button class="btn btn-secondary" onclick="app.callCustomer('${job.phone}')">
                    <i class="fas fa-phone"></i>
                    Call
                </button>
                <button class="btn btn-secondary" onclick="app.openMaps('${job.address}')">
                    <i class="fas fa-map-marker-alt"></i>
                    Navigate
                </button>
            `;
        } else if (this.currentTab === 'deliveries') {
            actionButtons = `
                <button class="btn btn-success" onclick="app.markAsDelivered('${job.id}')">
                    <i class="fas fa-check"></i>
                    Mark Delivered
                </button>
                <button class="btn btn-primary" onclick="app.capturePhoto('${job.id}', 'after')">
                    <i class="fas fa-camera"></i>
                    Take Photo
                </button>
                <button class="btn btn-secondary" onclick="app.callCustomer('${job.phone}')">
                    <i class="fas fa-phone"></i>
                    Call
                </button>
                <button class="btn btn-secondary" onclick="app.openMaps('${job.address}')">
                    <i class="fas fa-map-marker-alt"></i>
                    Navigate
                </button>
            `;
        } else {
            actionButtons = `
                <button class="btn btn-secondary" onclick="app.showJobDetails('${job.id}')">
                    <i class="fas fa-eye"></i>
                    View Details
                </button>
                <button class="btn btn-secondary" onclick="app.callCustomer('${job.phone}')">
                    <i class="fas fa-phone"></i>
                    Call
                </button>
            `;
        }

        return `
            <div class="job-card" onclick="app.showJobDetails('${job.id}')">
                <div class="job-header">
                    <span class="job-id">#${job.id}</span>
                    <div class="header-badges">
                        <span class="job-priority ${priorityClass}">${job.priority}</span>
                        <span class="status-badge ${statusClass}">${job.status}</span>
                    </div>
                </div>
                
                <h3 class="job-title">${job.title}</h3>
                
                <div class="job-details">
                    <div class="detail-item">
                        <i class="fas fa-user"></i>
                        <span>${job.customer}</span>
                    </div>
                    
                    <div class="detail-item vehicle-summary">
                        <div class="vehicle-summary-item">
                            <i class="fas fa-car"></i>
                            <span class="reg-number">${job.regNo || 'No Reg'}</span>
                        </div>
                        <div class="vehicle-summary-item">
                            <i class="fas fa-cog"></i>
                            <span>${job.quantity} wheels</span>
                        </div>
                    </div>
                    
                    <div class="detail-item job-address">
                        <i class="fas fa-map-marker-alt"></i>
                        <span>${job.address}</span>
                    </div>
                </div>
                
                <div class="job-actions" onclick="event.stopPropagation()">
                    ${actionButtons}
                </div>
            </div>
        `;
    }

    renderEmptyState() {
        return `
            <div class="empty-state">
                <i class="fas fa-inbox"></i>
                <h3>No jobs found</h3>
                <p>There are no jobs in this category at the moment.</p>
            </div>
        `;
    }

    markAsCollected(jobId) {
        const job = this.jobs.collections.find(j => j.id === jobId);
        if (!job) return;

        job.status = 'collected';
        
        // Move from collections to deliveries
        this.jobs.collections = this.jobs.collections.filter(j => j.id !== jobId);
        this.jobs.deliveries.push(job);
        
        this.saveJobs();
        this.renderCurrentTab();
        this.updateBadges();
        
        this.showToast(`Job #${jobId} marked as collected`, 'success');
    }

    markAsDelivered(jobId) {
        const job = this.jobs.deliveries.find(j => j.id === jobId);
        if (!job) return;

        job.status = 'delivered';
        job.completedAt = new Date().toISOString();
        
        // Move from deliveries to completed
        this.jobs.deliveries = this.jobs.deliveries.filter(j => j.id !== jobId);
        this.jobs.completed.push(job);
        
        this.saveJobs();
        this.renderCurrentTab();
        this.updateBadges();
        
        this.showToast(`Job #${jobId} marked as delivered`, 'success');
    }

    async capturePhoto(jobId, type) {
        try {
            this.currentJobId = jobId;
            const modal = document.getElementById('camera-modal');
            const video = document.getElementById('camera-video');
            
            // Request camera access
            this.cameraStream = await navigator.mediaDevices.getUserMedia({ 
                video: { 
                    width: { ideal: 1280 },
                    height: { ideal: 720 },
                    facingMode: 'environment' // Back camera on mobile
                } 
            });
            
            video.srcObject = this.cameraStream;
            modal.classList.add('active');
            
        } catch (error) {
            console.error('Error accessing camera:', error);
            this.showToast('Unable to access camera. Please check permissions.', 'error');
        }
    }

    capturePhoto() {
        const video = document.getElementById('camera-video');
        const canvas = document.getElementById('camera-canvas');
        const context = canvas.getContext('2d');
        
        // Set canvas dimensions to match video
        canvas.width = video.videoWidth;
        canvas.height = video.videoHeight;
        
        // Draw video frame to canvas
        context.drawImage(video, 0, 0);
        
        // Convert to base64
        const photoData = canvas.toDataURL('image/jpeg', 0.8);
        
        // Save photo (in a real app, upload to server)
        this.savePhoto(this.currentJobId, photoData);
        
        // Close camera modal
        this.closeCameraModal();
        
        this.showToast('Photo captured successfully', 'success');
    }

    savePhoto(jobId, photoData) {
        // In a real app, this would upload to a server
        const photos = JSON.parse(localStorage.getItem('jobPhotos') || '{}');
        if (!photos[jobId]) {
            photos[jobId] = [];
        }
        photos[jobId].push({
            timestamp: new Date().toISOString(),
            data: photoData
        });
        localStorage.setItem('jobPhotos', JSON.stringify(photos));
    }

    closeCameraModal() {
        const modal = document.getElementById('camera-modal');
        modal.classList.remove('active');
        
        if (this.cameraStream) {
            this.cameraStream.getTracks().forEach(track => track.stop());
            this.cameraStream = null;
        }
    }

    showJobDetails(jobId) {
        const job = this.findJobById(jobId);
        if (!job) return;

        const modal = document.getElementById('job-modal');
        const title = document.getElementById('modal-title');
        const body = document.getElementById('modal-body');

        title.textContent = `Job Details`;
        
        const completedDate = job.completedAt ? 
            new Date(job.completedAt).toLocaleDateString('en-GB', {
                weekday: 'long',
                year: 'numeric',
                month: 'long',
                day: 'numeric'
            }) : 'N/A';

        const priorityClass = `priority-${job.priority}`;
        const statusClass = `status-${job.status}`;

        // Generate action buttons based on current tab
        let quickActions = '';
        if (this.currentTab === 'collections') {
            quickActions = `
                <div class="quick-actions">
                    <button class="quick-action-btn primary" onclick="app.markAsCollected('${job.id}'); app.closeJobModal();">
                        <i class="fas fa-check-circle"></i>
                        <span>Mark as Collected</span>
                    </button>
                    <button class="quick-action-btn secondary" onclick="app.capturePhoto('${job.id}', 'before')">
                        <i class="fas fa-camera"></i>
                        <span>Take Photo</span>
                    </button>
                </div>
            `;
        } else if (this.currentTab === 'deliveries') {
            quickActions = `
                <div class="quick-actions">
                    <button class="quick-action-btn primary" onclick="app.markAsDelivered('${job.id}'); app.closeJobModal();">
                        <i class="fas fa-check-circle"></i>
                        <span>Mark as Delivered</span>
                    </button>
                    <button class="quick-action-btn secondary" onclick="app.capturePhoto('${job.id}', 'after')">
                        <i class="fas fa-camera"></i>
                        <span>Take Photo</span>
                    </button>
                </div>
            `;
        }

        body.innerHTML = `
            <div class="job-detail-view">
                <div class="job-detail-header">
                    <div class="job-detail-left">
                        <div class="job-detail-id">#${job.id}</div>
                    </div>
                    <div class="job-detail-badges">
                        <span class="job-priority ${priorityClass}">${job.priority}</span>
                        <span class="status-badge ${statusClass}">${job.status}</span>
                    </div>
                </div>
                
                <h3 class="job-detail-title">${job.title}</h3>
                
                ${quickActions}
                
                <div class="job-detail-grid">
                    <div class="detail-card">
                        <div class="detail-card-icon">
                            <i class="fas fa-user"></i>
                        </div>
                        <div class="detail-card-content">
                            <span class="detail-label">Customer</span>
                            <span class="detail-value">${job.customer}</span>
                        </div>
                    </div>
                    
                    <div class="detail-card vehicle-info-card">
                        <div class="vehicle-info-grid">
                            <div class="vehicle-info-item">
                                <div class="detail-card-icon">
                                    <i class="fas fa-car"></i>
                                </div>
                                <div class="detail-card-content">
                                    <span class="detail-label">Registration</span>
                                    <span class="detail-value reg-number">${job.regNo || 'No Reg'}</span>
                                </div>
                            </div>
                            <div class="vehicle-info-item">
                                <div class="detail-card-icon">
                                    <i class="fas fa-hashtag"></i>
                                </div>
                                <div class="detail-card-content">
                                    <span class="detail-label">Quantity</span>
                                    <span class="detail-value">${job.quantity} wheels</span>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="detail-card address-card">
                        <div class="detail-card-icon">
                            <i class="fas fa-map-marker-alt"></i>
                        </div>
                        <div class="detail-card-content">
                            <span class="detail-label">Address</span>
                            <span class="detail-value">${job.address}</span>
                        </div>
                        <button class="detail-action-btn" onclick="app.openMaps('${job.address}')">
                            <i class="fas fa-directions"></i>
                        </button>
                    </div>
                    
                    <div class="detail-card">
                        <div class="detail-card-icon">
                            <i class="fas fa-phone"></i>
                        </div>
                        <div class="detail-card-content">
                            <span class="detail-label">Phone</span>
                            <span class="detail-value">${job.phone}</span>
                        </div>
                        <button class="detail-action-btn" onclick="app.callCustomer('${job.phone}')">
                            <i class="fas fa-phone"></i>
                        </button>
                    </div>
                    
                    <div class="detail-card">
                        <div class="detail-card-icon">
                            <i class="fas fa-cog"></i>
                        </div>
                        <div class="detail-card-content">
                            <span class="detail-label">Service</span>
                            <span class="detail-value">${job.service}</span>
                        </div>
                    </div>
                    
                    
                    ${job.completedAt ? `
                    <div class="detail-card">
                        <div class="detail-card-icon">
                            <i class="fas fa-calendar-check"></i>
                        </div>
                        <div class="detail-card-content">
                            <span class="detail-label">Completed</span>
                            <span class="detail-value">${completedDate}</span>
                        </div>
                    </div>
                    ` : ''}
                    
                    ${job.notes ? `
                    <div class="detail-card notes-card">
                        <div class="detail-card-icon">
                            <i class="fas fa-sticky-note"></i>
                        </div>
                        <div class="detail-card-content">
                            <span class="detail-label">Notes</span>
                            <span class="detail-value">${job.notes}</span>
                        </div>
                    </div>
                    ` : ''}
                </div>
            </div>
        `;

        modal.classList.add('active');
    }

    findJobById(jobId) {
        for (const category of Object.values(this.jobs)) {
            const job = category.find(j => j.id === jobId);
            if (job) return job;
        }
        return null;
    }

    closeJobModal() {
        document.getElementById('job-modal').classList.remove('active');
    }

    closeAllModals() {
        document.querySelectorAll('.modal').forEach(modal => {
            modal.classList.remove('active');
        });
        
        if (this.cameraStream) {
            this.cameraStream.getTracks().forEach(track => track.stop());
            this.cameraStream = null;
        }
    }

    openMaps(address) {
        const encodedAddress = encodeURIComponent(address);
        const url = `https://www.google.com/maps/search/?api=1&query=${encodedAddress}`;
        window.open(url, '_blank');
    }

    callCustomer(phone) {
        window.location.href = `tel:${phone}`;
    }

    updateBadges() {
        document.getElementById('collections-badge').textContent = this.jobs.collections.length;
        document.getElementById('deliveries-badge').textContent = this.jobs.deliveries.length;
        document.getElementById('completed-badge').textContent = this.jobs.completed.length;
    }

    async refreshJobs() {
        this.showToast('Refreshing jobs...', 'info');
        
        // Simulate API call
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // In a real app, this would fetch from the server
        this.renderCurrentTab();
        this.updateBadges();
        
        this.showToast('Jobs refreshed', 'success');
    }

    showToast(message, type = 'info') {
        const container = document.getElementById('toast-container');
        const toast = document.createElement('div');
        toast.className = `toast ${type}`;
        toast.textContent = message;
        
        container.appendChild(toast);
        
        // Trigger animation
        setTimeout(() => toast.classList.add('show'), 100);
        
        // Remove after 3 seconds
        setTimeout(() => {
            toast.classList.remove('show');
            setTimeout(() => container.removeChild(toast), 300);
        }, 3000);
    }
}

// Global functions for inline event handlers
function closeJobModal() {
    app.closeJobModal();
}

function closeCameraModal() {
    app.closeCameraModal();
}

function capturePhoto() {
    app.capturePhoto();
}

function closeNewJobModal() {
    // Implementation for new job modal if needed
}

function refreshJobs() {
    app.refreshJobs();
}

// Initialize the app
const app = new VanDriverApp();